# -*- coding: utf-8 -*-

import datetime
import distutils
import json
import logging
import os
import re
import urllib2
from collections import namedtuple, OrderedDict
from shutil import copyfile

from sandbox import sdk2

import sandbox.common.types.task as ctt
from sandbox.common.errors import TaskError

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk.channel import channel

import sandbox.projects.resource_types as rt

from sandbox.projects.common.dolbilka import DolbilkaExecutorRequestsLimit, DolbilkaMaximumSimultaneousRequests

from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
import sandbox.projects.release_machine.components.all as rmc

import sandbox.projects.geosuggest.component as geo_suggest
from sandbox.projects.geosuggest.common import nanny_api
from sandbox.projects.geosuggest.common.report import EmailBuilder
from sandbox.projects.geosuggest.common.utils import ensure_task_succeeded
from sandbox.projects.geosuggest.GeoSuggestTestFunctional import GeoSuggestTestFunctional
from sandbox.projects.geosuggest.GeoSuggestTestFunctionalDiff import GeoSuggestTestFunctionalDiff
from sandbox.projects.geosuggest.GeoSuggestTestPerformanceParallel import GeoSuggestTestPerformanceParallel
from sandbox.projects.geosuggest.resources import (
    GEO_SUGGEST_DATA,
    GEO_SUGGEST_MATRIXNET_MODEL,
    GEO_SUGGEST_PRIEMKA_REPORT,
    GEO_SUGGEST_TEST_FUNCTIONAL_REPORT,
    GEO_SUGGEST_WEBDAEMON,
    GEO_SUGGEST_WEBDAEMON_CONFIG,
    GEO_SUGGEST_WEBDAEMON_PLAN,
    MAPS_GEO_SUGGEST_DATA_BUILDER
)
from sandbox.projects.geosuggest.component import (
    DEFAULT_GEO_SUGGEST_START_TIMEOUT,
    DEFAULT_GEO_SUGGEST_SHUTDOWN_TIMEOUT
)
from sandbox.projects.geosuggest.tests import ACCEPTANCE_QUALITY_CONDITIONS

ActiveNannyResourceIds = namedtuple('ActiveNannyResourceIds', ['daemon', 'data'])


class GeoSuggestAcceptance(sdk2.Task):
    """
        Test suggest metrics
    """
    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client,
                        PipEnvironment('yandex-yt')]
        client_tags = task_env.TaskTags.startrek_client
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.task.Parameters):
        with sdk2.parameters.Group('Launch parameters') as launch_params:
            geosuggestd_resource_id = sdk2.parameters.Resource(
                'Geo suggest daemon, see https://nda.ya.ru/3RbPHV',
                resource_type=GEO_SUGGEST_WEBDAEMON,
                required=True)
            geosuggest_aux_formula = sdk2.parameters.Resource(
                'Geo suggest aux formula resource id (experimental)',
                resource_type=GEO_SUGGEST_MATRIXNET_MODEL)
            geosuggest_aux_formulas_fml_ids = sdk2.parameters.List(
                'Geo suggest aux formulas fml-ids list (experimental)',
                default=None,
                required=False,
                value_type=sdk2.parameters.String)
            geosuggest_aux_experiments = sdk2.parameters.List(
                'Geo suggest aux experiments list (experimental)',
                default=None,
                required=False,
                value_type=sdk2.parameters.String)
            geosuggest_custom_flags = sdk2.parameters.Dict(
                'Custom flags list (experimental)',
                default=None,
                required=False,
                value_type=sdk2.parameters.String)
            geosuggest_data_resource_id = sdk2.parameters.Resource(
                'Geo suggest data, see https://nda.ya.ru/3RbPHf',
                resource_type=GEO_SUGGEST_DATA,
                required=True)
            geosuggest_config_resource_id = sdk2.parameters.Resource(
                'Geo suggest config, by default will be taken from data',
                resource_type=GEO_SUGGEST_WEBDAEMON_CONFIG)
            geosuggest_start_timeout = sdk2.parameters.Integer(
                'Geo suggest start timeout in seconds',
                default=DEFAULT_GEO_SUGGEST_START_TIMEOUT)
            geosuggest_shutdown_timeout = sdk2.parameters.Integer(
                'Geo suggest shutdown timeout in seconds',
                default=DEFAULT_GEO_SUGGEST_SHUTDOWN_TIMEOUT)
            isolated_mode = sdk2.parameters.Bool(
                'Isolated mode: do not use external services',
                default=False)
        with sdk2.parameters.Group('Data builder') as data_builder_params:
            data_builder = sdk2.parameters.Resource(
                'Geo suggest data builder executable files',
                resource_type=MAPS_GEO_SUGGEST_DATA_BUILDER,
                required=True)
        with sdk2.parameters.Group('Performance test') as performance_test_params:
            geosuggest_run_performance_tests = sdk2.parameters.Bool(
                'Run performance tests',
                default=True)
            geosuggest_dolbilka_plan_resource_id = sdk2.parameters.Resource(
                'Geo suggest dolbilka plan, see https://nda.ya.ru/3RbRcr',
                resource_type=GEO_SUGGEST_WEBDAEMON_PLAN,
                required=False)
        with sdk2.parameters.Group('Functional test') as functional_test_params:
            with sdk2.parameters.String('Old tests results') as geosuggest_stable_result_parameter:
                geosuggest_stable_result_parameter.values.production = 'Use running production service'
                geosuggest_stable_result_parameter.values.external = 'External daemon'
                geosuggest_stable_result_parameter.values.another_shard = 'Another shard and daemon'
                geosuggest_stable_result_parameter.values.stable_shard = 'Auto pick last stable shard'
                geosuggest_stable_result_parameter.values.task_logs = 'TASK_LOGS'
                geosuggest_stable_result_parameter.values.production_snapshot = \
                    geosuggest_stable_result_parameter.Value('The same data and the same daemon, as in production', default=True)
            with geosuggest_stable_result_parameter.value.external:
                geosuggest_external_daemon_url = sdk2.parameters.String(
                    'External geosuggest daemon url',
                    default='http://suggest-maps.yandex.{domain}/suggest-geo')
            with geosuggest_stable_result_parameter.value.another_shard:
                geosuggest_another_daemon_resource_id = sdk2.parameters.Resource(
                    'Another daemon',
                    resource_type=GEO_SUGGEST_WEBDAEMON,
                    required=True)
                geosuggest_another_aux_formula = sdk2.parameters.Resource(
                    'Another aux formula resource id (experimental)',
                    resource_type=GEO_SUGGEST_MATRIXNET_MODEL)
                geosuggest_another_aux_formulas_fml_ids = sdk2.parameters.List(
                    'Another aux formulas fml-ids list (experimental)',
                    default=None,
                    required=False,
                    value_type=sdk2.parameters.String)
                geosuggest_another_aux_experiments = sdk2.parameters.List(
                    'Another aux experiments list (experimental)',
                    default=None,
                    required=False,
                    value_type=sdk2.parameters.String)
                geosuggest_another_custom_flags = sdk2.parameters.Dict(
                    'Another custom flags list (experimental)',
                    default=None,
                    required=False,
                    value_type=sdk2.parameters.String)
                geosuggest_another_data_resource_id = sdk2.parameters.Resource(
                    'Another data',
                    resource_type=GEO_SUGGEST_DATA,
                    required=True)
            with geosuggest_stable_result_parameter.value.task_logs:
                geosuggest_task_logs = sdk2.parameters.Resource(
                    'Ready functional tests report',
                    resource_type=rt.TASK_LOGS)
            geosuggest_autotest_arguments = sdk2.parameters.String(
                'Additional arguments to autotest.py',
                default='')
            geosuggest_use_personal_mocks = sdk2.parameters.Bool(
                GeoSuggestTestFunctional.GeoSuggestUsePersonalMocksParameter.description,
                default=False)
            geosuggest_path_to_autotest = sdk2.parameters.String(
                'Path to autotest.py inside data builder pack',
                default='autotest')
        with sdk2.parameters.Group('Report') as report_params:
            geosuggest_emails_to = sdk2.parameters.String(
                'Send email with results to',
                default='')
            component_name = sdk2.parameters.String(
                'Component name',
                default='')
            branch = sdk2.parameters.Integer(
                'Branch',
                required=False)
        with sdk2.parameters.Group('Nanny token') as nanny_token_params:
            geosuggest_sandbox_vault_owner = sdk2.parameters.String('Nanny token owner')
            geosuggest_sandbox_vault_nanny_token_name = sdk2.parameters.String('Nanny token name', default='nanny_token')

    class Context(sdk2.Task.Context):
        active_nanny_resources = {}
        subtask_diff = None
        geosuggest_autodeploy = None
        subtask_performance = {}
        subtask_new_tests = {}
        subtask_old_tests = {}
        FINAL_DIFF_REPORT_RESOURCE = None
        FINAL_PRIEMKA_REPORT = None

    def write_to_startrek(self, comment):
        if not self.Parameters.component_name or self.Parameters.branch is None:
            logging.info('No branch or component name. Not posting comment to startrek.')
            return
        startrek_token = sdk2.Vault.data('robot-geosearch',
                                         'robot_geosearch_startrek_token')
        try:
            startrek_helper = STHelper(startrek_token,
                                       useragent='robot-geosearch')
            c_info = rmc.get_component(self.Parameters.component_name)
            startrek_helper.comment(self.Parameters.branch, comment, c_info)
        except Exception:
            logging.exception('Could not post comment to startrek')
            logging.info(comment)

    def get_nanny_token(self):
        if self.Parameters.geosuggest_sandbox_vault_owner:
            return sdk2.Vault.data(self.Parameters.geosuggest_sandbox_vault_owner,
                                   self.Parameters.geosuggest_sandbox_vault_nanny_token_name)
        else:
            return sdk2.Vault.data(self.Parameters.geosuggest_sandbox_vault_nanny_token_name)

    def get_active_nanny_resources(self):
        nanny_client = nanny_api.get_nanny_client(self.get_nanny_token())
        snapshot_resource_ids = nanny_api.get_active_snapshot_resource_ids(nanny_client, 'suggest_maps_yp')
        if snapshot_resource_ids is None:
            raise TaskError('Unable to get active state from Nanny')

        # for debugging
        for resource_type in [GEO_SUGGEST_WEBDAEMON, GEO_SUGGEST_DATA]:
            name = str(resource_type)
            logging.info('Active %s: %s', name, snapshot_resource_ids.get(name))
            self.Context.active_nanny_resources[name] = snapshot_resource_ids.get(name)

        return ActiveNannyResourceIds(
            daemon=snapshot_resource_ids[str(GEO_SUGGEST_WEBDAEMON)],
            data=snapshot_resource_ids[str(GEO_SUGGEST_DATA)],
        )

    def copy_from_logs(self, task_id, files_to_new_path):
        resource_data = sdk2.ResourceData(sdk2.Resource.find(resource_type=rt.TASK_LOGS, task=sdk2.Task[task_id]).first())
        path = str(resource_data.path)
        for filename in files_to_new_path:
            fullname = os.path.join(path, filename)
            if os.path.isfile(fullname):
                copyfile(
                    fullname,
                    files_to_new_path[filename]
                )
            else:
                distutils.dir_util.copy_tree(
                    fullname,
                    files_to_new_path[filename]
                )

    def prepare_footer(self, pack):
        with open(os.path.join(pack, "autotest_diff.out.txt"), 'r') as diff_file:
            footer = GeoSuggestTestFunctionalDiff.get_footer_by_content(diff_file.read(), limit=800)
            with open(os.path.join(pack, "footer.json"), 'w') as footer_file:
                json.dump(footer, footer_file, indent=4)

    def retry_subtask(self, ctx_obj, task_type, description, input_parameters, retries, is_sdk2=False):
        id = ctx_obj.get("id", None)
        retries_remain = ctx_obj.get("retries_remain", retries)
        need_retry = True
        if id is not None:
            status = sdk2.Task[id].status
            logging.info('Task %s, status %s' % (id, status))
            if status in (ctt.Status.Group.FINISH + ctt.Status.Group.BREAK) and (status not in ctt.Status.Group.SUCCEED):
                need_retry = True
            else:
                need_retry = False
            logging.info('Need retry %s, remain %s' % (need_retry, retries_remain))
        if need_retry and retries_remain > 0:
            ctx_obj['retries_remain'] = retries_remain - 1
            if is_sdk2:
                id = task_type(self, description=description, **input_parameters).enqueue().id
            else:
                id = sdk2.Task[task_type](self, description=description, **input_parameters).enqueue().id
            ctx_obj['id'] = id
            logging.info('New task %s' % id)
        return id

    def need_to_wait_subtask(self, *ctx_objs):
        for ctx_obj in ctx_objs:
            if not ctx_obj:
                continue
            task = sdk2.Task[ctx_obj["id"]]

            if task.status in (ctt.Status.Group.FINISH + ctt.Status.Group.BREAK):
                # subtask succeeded
                if task.status in ctt.Status.Group.SUCCEED:
                    continue
                # subtask failed
                if ctx_obj["retries_remain"] == 0:
                    continue
            # subtask in progress or needs a retry.
            return True
        return False

    def get_tables_from_footer(self, footer):
        # footer example:
        # [
        #     // table item:
        #     {
        #         "helperName": "performance",
        #         "content":
        #         {
        #             "<h3>Performance</h3>":
        #             [
        #                 { "Suite": "ru", "Ok": 32, ...},
        #                 { "Suite": "en", "Ok": 46, ...},
        #             ]
        #         },
        #     },
        #     // text item:
        #     {
        #        "helperName": "diff",
        #        "content": "Tests for ...",
        #     },
        # ]
        for footer_item in footer:
            content = footer_item.get("content")
            if not content or not isinstance(content, dict):
                continue

            for _, lines in content.iteritems():
                if not lines or not isinstance(lines, list):
                    continue

                column_names = [name for name in lines[0]]
                rows = []
                for line in lines:
                    rows.append([line.get(name, "") for name in column_names])
                yield column_names, rows

    def send_email_report(self, daemon_resource_id, data_resource_id, is_passed, nanny_main_snapshot_id, nanny_experimental_snapshot_id):
        emails_to = self.Parameters.geosuggest_emails_to
        if not emails_to:
            return

        logging.info("Preparing email report...")

        daemon_resource = sdk2.Resource[daemon_resource_id]
        data_resource = sdk2.Resource[data_resource_id]
        daemon_task = daemon_resource.task
        data_task = data_resource.task

        report = EmailBuilder()
        report.add_paragraph("Привет!")
        report.add_space()
        if is_passed:
            report.add_paragraph("Новый шард прошёл приёмку:")
        else:
            report.add_paragraph("Новый шард не прошёл приёмку:")
        report.add_link("https://sandbox.yandex-team.ru/task/{0}/view".format(self.id), "{0}:{1}".format(self.type, self.id))
        report.add_space()
        if is_passed:
            if nanny_main_snapshot_id:
                report.add_paragraph("Снапшот (suggest_maps_yp):")
                report.add_link(
                    "https://nanny.yandex-team.ru/ui/#/services/catalog/suggest_maps_yp/runtime_attrs_history/{}/".format(nanny_main_snapshot_id),
                    "suggest_maps_yp:{0}".format(nanny_main_snapshot_id))
                report.add_space()
            if nanny_experimental_snapshot_id:
                report.add_paragraph("Снапшот (suggest_maps_priemka):")
                report.add_link(
                    "https://nanny.yandex-team.ru/ui/#/services/catalog/suggest_maps_priemka/runtime_attrs_history/{}/".format(nanny_experimental_snapshot_id),
                    "suggest_maps_yp:{0}".format(nanny_experimental_snapshot_id))
                report.add_space()
        else:
            report.add_paragraph("Нужно посмотреть на тесты глазами и исправить проблему, либо, если срабатывание ложное, а проблемы на самом деле нет, отправить шард в прод.")
            report.add_space()

        report.add_paragraph("Шард:")
        report.add_link(
            "https://sandbox.yandex-team.ru/task/{0}/view".format(data_task.id),
            "{0}:{1}".format(data_task.type, data_task.id))
        report.add_link(
            "https://sandbox.yandex-team.ru/resource/{0}/view".format(data_resource.id),
            "{0}:{1}".format(data_resource.type, data_resource.id))
        report.add_space()
        report.add_paragraph("Бинарь:")
        report.add_link(
            "https://sandbox.yandex-team.ru/task/{0}/view".format(daemon_task.id),
            "{0}:{1}".format(daemon_task.type, daemon_task.id), comment=daemon_task.description)
        report.add_link(
            "https://sandbox.yandex-team.ru/resource/{0}/view".format(daemon_resource.id),
            "{0}:{1}".format(daemon_resource.type, daemon_resource.id),
            comment=daemon_resource.description)
        report.add_space()

        try:
            performance_footer = self.get_performance_footer()[0]
            functional_footer = self.get_functional_footer()[0]
            for footer in [performance_footer, functional_footer]:
                for column_names, rows in self.get_tables_from_footer(footer):
                    report.add_table(column_names, rows)
                    report.add_space()
        except Exception:
            logging.exception("Failed to extract data from footers")

        report.add_signature()

        logging.info("Email report body:\n%s", report.get_html())

        result = "ok" if is_passed else "failed"
        date = datetime.date(1, 1, 1).today().strftime("%d.%m.%y")
        subject = "{0} {1} {2} #{3}".format(self.type, result, date, self.id)
        channel.sandbox.send_email(
            emails_to.split(","),
            None,
            subject,
            report.get_html(),
            "text/html",
            "utf-8",
        )
        logging.info("Email report sent to %s", emails_to)

    def release_daemon_and_data_to_nanny(self, daemon_resource_id, data_resource_id, is_stable_release):
        target_service_id = "suggest_maps_yp" if is_stable_release else "suggest_maps_priemka"
        try:
            logging.info("Releasing daemon #%s and data #%s to Nanny service '%s'...", daemon_resource_id, data_resource_id, target_service_id)
            daemon_resource = sdk2.Resource[daemon_resource_id]
            daemon_task = daemon_resource.task
            data_resource = sdk2.Resource[data_resource_id]
            data_task = data_resource.task

            resources_to_release = [
                {"task_type": daemon_task.type, "task_id": daemon_task.id, "resource_type": daemon_resource.type, "resource_id": daemon_resource.id},
                {"task_type": data_task.type, "task_id": data_task.id, "resource_type": data_resource.type, "resource_id": data_resource.id},
            ]

            release_title = "GEO_SUGGEST_DATA #{} + GEO_SUGGEST_WEBDAEMON #{} ({})".format(
                data_resource.id,
                daemon_resource.id,
                daemon_task.description)

            nanny_client = nanny_api.get_nanny_client(self.get_nanny_token())
            snapshot_id = nanny_api.update_service_resources(nanny_client, target_service_id, resources_to_release, release_title)
            if snapshot_id:
                logging.info("Release of daemon #%s and data #%s to Nanny service '%s' created, snapshot_id: '%s'.", daemon_resource_id, data_resource_id, target_service_id, snapshot_id)
                nanny_api.set_snapshot_to_prepared_state(nanny_client, target_service_id, snapshot_id)
                logging.info("Releasing daemon #%s and data #%s to Nanny service '%s' succeeded.", daemon_resource_id, data_resource_id, target_service_id)
            return snapshot_id

        except Exception:
            logging.exception("Releasing daemon #%s and data #%s to Nanny service '%s' failed.", daemon_resource_id, data_resource_id, target_service_id)
            return None

    def on_save(self):
        if self.Parameters.geosuggest_run_performance_tests and self.Parameters.geosuggest_dolbilka_plan_resource_id is None:
            self.Parameters.geosuggest_dolbilka_plan_resource_id = sdk2.Resource.find(
                resource_type=GEO_SUGGEST_WEBDAEMON_PLAN,
                attrs={'use_for_acceptance': 'true'}).order(-sdk2.Resource.id).first()

    def on_execute(self):
        with self.memoize_stage.ACCEPTANCE_START_COMMENT(commit_on_entrance=False):
            self.write_to_startrek('Acceptance task: https://sandbox.yandex-team.ru/task/{tid}/view'.format(tid=self.id))
        functional_test_context = {
            GeoSuggestTestFunctional.GeoSuggestUsePersonalMocksParameter.name: self.Parameters.geosuggest_use_personal_mocks,
            'geosuggest_config_resource_id': self.Parameters.geosuggest_config_resource_id.id if self.Parameters.geosuggest_config_resource_id else None,
            'geosuggest_start_timeout': self.Parameters.geosuggest_start_timeout,
            'geosuggest_shutdown_timeout': self.Parameters.geosuggest_shutdown_timeout,
            'data_builder': self.Parameters.data_builder.id,
            'geosuggest_autotest_arguments': self.Parameters.geosuggest_autotest_arguments,
            'geosuggestd_resource_id': self.Parameters.geosuggestd_resource_id.id,
            'geosuggest_data_resource_id': self.Parameters.geosuggest_data_resource_id.id,
            'geosuggest_dolbilka_plan_resource_id': self.Parameters.geosuggest_dolbilka_plan_resource_id.id,
            'isolated_mode': self.Parameters.isolated_mode,
            DolbilkaExecutorRequestsLimit.name: 500000,
            DolbilkaMaximumSimultaneousRequests.name: 50,
            'kill_timeout': self.Parameters.kill_timeout,
        }

        candidate_test_context = dict(functional_test_context.items() + {
            GeoSuggestTestFunctional.GeoSuggestAuxFormulaParameter.name: self.Parameters.geosuggest_aux_formula,
            GeoSuggestTestFunctional.GeoSuggestAuxFormulasParameter.name: self.Parameters.geosuggest_aux_formulas_fml_ids,
            GeoSuggestTestFunctional.GeoSuggestAuxExperimentsParameter.name: self.Parameters.geosuggest_aux_experiments,
            GeoSuggestTestFunctional.GeoSuggestCustomFlagsParameter.name: self.Parameters.geosuggest_custom_flags,
            GeoSuggestTestFunctional.GeoSuggestPathToAutotest.name: self.Parameters.geosuggest_path_to_autotest
        }.items())

        logging.info('Candidate test context: {}'.format(candidate_test_context))

        subtask_new_tests = self.retry_subtask(
            self.Context.subtask_new_tests,
            task_type=GeoSuggestTestFunctional.type,
            description='Test new shard (GEO_SUGGEST_WEBDAEMON #{}, GEO_SUGGEST_DATA #{})'.format(
                functional_test_context['geosuggestd_resource_id'],
                functional_test_context['geosuggest_data_resource_id']
            ),
            input_parameters=candidate_test_context,
            retries=3,
        )
        if self.Parameters.geosuggest_run_performance_tests:
            if self.Parameters.geosuggest_stable_result_parameter == 'another_shard':
                # use explicitly specified resources
                reference_daemon_id = self.Parameters.geosuggest_another_daemon_resource_id
                reference_data_id = self.Parameters.geosuggest_another_data_resource_id

                reference_aux_formula = self.Parameters.geosuggest_another_aux_formula
                reference_aux_formulas_fml_ids = self.Parameters.geosuggest_another_aux_formulas_fml_ids
                reference_aux_experiments = self.Parameters.geosuggest_another_aux_experiments
                reference_custom_flags = self.Parameters.geosuggest_another_custom_flags
            else:
                # take active snapshot from production
                active_resource_ids = self.get_active_nanny_resources()
                reference_daemon_id = active_resource_ids.daemon
                reference_data_id = active_resource_ids.data

                reference_aux_formula = None
                reference_aux_formulas_fml_ids = None
                reference_aux_experiments = None
                reference_custom_flags = None

            test_daemon_id = self.Parameters.geosuggestd_resource_id
            test_data_id = self.Parameters.geosuggest_data_resource_id
            test_aux_formula = self.Parameters.geosuggest_aux_formula
            test_aux_formulas_fml_ids = self.Parameters.geosuggest_aux_formulas_fml_ids
            test_aux_experiments = self.Parameters.geosuggest_aux_experiments
            test_custom_flags = self.Parameters.geosuggest_custom_flags
            subtask_performance = self.retry_subtask(
                self.Context.subtask_performance,
                task_type=GeoSuggestTestPerformanceParallel,
                description='Test performance for priemka (GEO_SUGGEST_WEBDAEMON #{}, GEO_SUGGEST_DATA #{})'.format(test_daemon_id, test_data_id),
                input_parameters={
                    'reference_daemon': reference_daemon_id,
                    'reference_data': reference_data_id,
                    'reference_aux_formula': reference_aux_formula,
                    'reference_aux_formulas_fml_ids': reference_aux_formulas_fml_ids,
                    'reference_aux_experiments': reference_aux_experiments,
                    'reference_custom_flags': reference_custom_flags,
                    'test_daemon': test_daemon_id,
                    'test_data': test_data_id,
                    'test_aux_formula': test_aux_formula,
                    'test_aux_formulas_fml_ids': test_aux_formulas_fml_ids,
                    'test_aux_experiments': test_aux_experiments,
                    'test_custom_flags': test_custom_flags,
                    'dolbilo_plan': self.Parameters.geosuggest_dolbilka_plan_resource_id,
                    'dolbilka_executor_sessions': 5,
                    'dolbilka_executor_requests_limit': 100000,
                    'enable_lunapark': False,
                    'kill_timeout': self.Parameters.kill_timeout,
                },
                retries=3,
                is_sdk2=True,
            )

        subtask_old_tests = None
        tests_old_logs = None
        tests_old_way = self.Parameters.geosuggest_stable_result_parameter
        if tests_old_way == 'production':
            subtask_old_tests = self.retry_subtask(
                self.Context.subtask_old_tests,
                task_type=GeoSuggestTestFunctional.type,
                description='Getting test results from production',
                input_parameters=dict(functional_test_context.items() + {
                    GeoSuggestTestFunctional.GeoSuggestExternalDaemonUrl.name: 'http://suggest-maps.yandex.{domain}/suggest-geo',
                    GeoSuggestTestFunctional.GeoSuggestPathToAutotest.name: self.Parameters.geosuggest_path_to_autotest,
                    'kill_timeout': self.Parameters.kill_timeout,
                }.items()),
                retries=3,
            )
        elif tests_old_way == 'external':
            subtask_old_tests = self.retry_subtask(
                self.Context.subtask_old_tests,
                task_type=GeoSuggestTestFunctional.type,
                description='Getting test results from external daemon',
                input_parameters=dict(functional_test_context.items() + {
                    GeoSuggestTestFunctional.GeoSuggestExternalDaemonUrl.name: self.Parameters.geosuggest_external_daemon_url,
                    GeoSuggestTestFunctional.GeoSuggestPathToAutotest.name: self.Parameters.geosuggest_path_to_autotest,
                    'kill_timeout': self.Parameters.kill_timeout,
                }.items()),
                retries=3,
            )
        elif tests_old_way == 'another_shard':
            subtask_old_tests = self.retry_subtask(
                self.Context.subtask_old_tests,
                task_type=GeoSuggestTestFunctional.type,
                description='Getting tests from another daemon+shard',
                input_parameters=dict(functional_test_context.items() + {
                    GeoSuggestTestFunctional.GeoSuggestAutotestArguments.name: self.Parameters.geosuggest_autotest_arguments,
                    GeoSuggestTestFunctional.GeoSuggestPathToAutotest.name: self.Parameters.geosuggest_path_to_autotest,
                    geo_suggest.GeoSuggestDaemonParameter.name: self.Parameters.geosuggest_another_daemon_resource_id.id,
                    geo_suggest.GeoSuggestDataParameter.name: self.Parameters.geosuggest_another_data_resource_id.id,
                    'kill_timeout': self.Parameters.kill_timeout,
                    GeoSuggestTestFunctional.GeoSuggestAuxFormulaParameter.name: self.Parameters.geosuggest_another_aux_formula,
                    GeoSuggestTestFunctional.GeoSuggestAuxFormulasParameter.name: self.Parameters.geosuggest_another_aux_formulas_fml_ids,
                    GeoSuggestTestFunctional.GeoSuggestAuxExperimentsParameter.name: self.Parameters.geosuggest_another_aux_experiments,
                    GeoSuggestTestFunctional.GeoSuggestCustomFlagsParameter.name: self.Parameters.geosuggest_another_custom_flags,
                }.items()),
                retries=3,
            )
        elif tests_old_way == 'stable_shard':
            stable_shard = sdk2.Resource.find(GEO_SUGGEST_DATA).order(-sdk2.Resource.id).first().id
            subtask_old_tests = self.retry_subtask(
                self.Context.subtask_old_tests,
                task_type=GeoSuggestTestFunctional.type,
                description='Getting tests from last stable shard',
                input_parameters=dict(functional_test_context.items() + {
                    GeoSuggestTestFunctional.GeoSuggestAutotestArguments.name: self.Parameters.geosuggest_autotest_arguments,
                    GeoSuggestTestFunctional.GeoSuggestPathToAutotest.name: self.Parameters.geosuggest_path_to_autotest,
                    geo_suggest.GeoSuggestDataParameter.name: stable_shard,
                    'kill_timeout': self.Parameters.kill_timeout,
                }.items()),
                retries=3,
            )
        elif tests_old_way == 'task_logs':
            tests_old_logs = self.Parameters.geosuggest_task_logs
        elif tests_old_way == 'production_snapshot':
            active_resource_ids = self.get_active_nanny_resources()
            subtask_old_tests = self.retry_subtask(
                self.Context.subtask_old_tests,
                task_type=GeoSuggestTestFunctional.type,
                description='Reference test results from production snapshot (GEO_SUGGEST_WEBDAEMON #{}, GEO_SUGGEST_DATA #{})'.format(
                    active_resource_ids.daemon,
                    active_resource_ids.data
                ),
                input_parameters=dict(functional_test_context.items() + {
                    GeoSuggestTestFunctional.GeoSuggestAutotestArguments.name: self.Parameters.geosuggest_autotest_arguments,
                    GeoSuggestTestFunctional.GeoSuggestPathToAutotest.name: self.Parameters.geosuggest_path_to_autotest,
                    geo_suggest.GeoSuggestDaemonParameter.name: active_resource_ids.daemon,
                    geo_suggest.GeoSuggestDataParameter.name: active_resource_ids.data,
                    'kill_timeout': self.Parameters.kill_timeout,
                }.items()),
                retries=3,
            )

        tasks = [task for task in (subtask_new_tests, subtask_old_tests) if task is not None]

        # with self.memoize_stage.exec_wait1:
        if self.need_to_wait_subtask(self.Context.subtask_new_tests, self.Context.subtask_old_tests):
            raise sdk2.WaitTask(tasks, ctt.Status.Group.FINISH + ctt.Status.Group.BREAK, wait_all=True)

        tests_new_logs = None
        if subtask_new_tests is not None:
            ensure_task_succeeded(subtask_new_tests, "new shard tests subtask")
            tests_new_logs = sdk2.Resource.find(resource_type=GEO_SUGGEST_TEST_FUNCTIONAL_REPORT, task=sdk2.Task[subtask_new_tests]).first().id

        if subtask_old_tests is not None:
            ensure_task_succeeded(subtask_old_tests, "old shard tests subtask")
            tests_old_logs = sdk2.Resource.find(resource_type=GEO_SUGGEST_TEST_FUNCTIONAL_REPORT, task=sdk2.Task[subtask_old_tests]).first().id

        with self.memoize_stage.exec_diff:
            self.Context.subtask_diff = sdk2.Task[GeoSuggestTestFunctionalDiff.type](
                self,
                description='Diff of new and "old" shards',
                **{
                    'data_builder': self.Parameters.data_builder.id,
                    GeoSuggestTestFunctionalDiff.GeoSuggestAutotestReportOld.name: tests_old_logs,
                    GeoSuggestTestFunctionalDiff.GeoSuggestAutotestReportNew.name: tests_new_logs,
                }
            ).enqueue().id
        with self.memoize_stage.exec_wait2:
            raise sdk2.WaitTask([self.Context.subtask_diff], ctt.Status.Group.FINISH + ctt.Status.Group.BREAK, wait_all=True)
        ensure_task_succeeded(self.Context.subtask_diff, "diff tests subtask")

        with self.memoize_stage.wait_performance:
            if self.Parameters.geosuggest_run_performance_tests:
                raise sdk2.WaitTask([subtask_performance], ctt.Status.Group.FINISH + ctt.Status.Group.BREAK, wait_all=True)

        self.Context.FINAL_DIFF_REPORT_RESOURCE = sdk2.Resource.find(resource_type=rt.TASK_LOGS, task=sdk2.Task[self.Context.subtask_diff]).first().id

        pack = str(self.path('pack'))
        distutils.dir_util.mkpath(pack)
        self.copy_from_logs(self.Context.subtask_diff, {
            "autotest_diff.out.txt": os.path.join(pack, "autotest_diff.out.txt")
        })
        self.copy_from_logs(subtask_old_tests, {
            "autotest_report.json": os.path.join(pack, "autotest_report.old.json")
        })
        self.copy_from_logs(subtask_new_tests, {
            "autotest_report.json": os.path.join(pack, "autotest_report.new.json")
        })
        try:
            self.prepare_footer(pack)
        except:
            # we still can render footer later
            pass
        resource = sdk2.Resource[GEO_SUGGEST_PRIEMKA_REPORT](self, 'Geo suggest priemka report', pack)
        sdk2.ResourceData(resource).ready()
        self.Context.FINAL_PRIEMKA_REPORT = resource.id

        is_performance_passed = self.get_performance_footer()[1]
        is_quality_passed = self.get_functional_footer()[1]
        logging.info("Performance tests passed: %s", is_performance_passed)
        logging.info("Quality tests passed: %s", is_quality_passed)

        is_passed = is_performance_passed and is_quality_passed
        self.Context.geosuggest_autodeploy = is_passed

        daemon_resource_id = self.Parameters.geosuggestd_resource_id
        data_resource_id = self.Parameters.geosuggest_data_resource_id

        nanny_main_snapshot_id = None
        nanny_experimental_snapshot_id = None
        try:
            self.send_email_report(daemon_resource_id, data_resource_id, is_passed, nanny_main_snapshot_id, nanny_experimental_snapshot_id)
        except Exception:
            logging.exception("Send email report failed")
        if self.Parameters.component_name and self.Parameters.branch is not None:
            try:
                self.write_startrek_comments()
            except Exception:
                logging.info('Failed to write comments to Startrek')
                logging.exception('Failed to write comments to Startrek')

        if self.parent is not None:
            if self.Context.geosuggest_autodeploy and self.parent.Context.production_build:
                self.server.release(task_id=self.parent.Context.postprocessing_task,
                                    type='stable',
                                    subject='Regular geosuggest index release',
                                    comments='Released due to acceptance results')

    @classmethod
    def extract_performance_results(cls, task_id):
        stats = sdk2.Task[task_id].Context.stats
        results = {
            'max_requests_per_sec': stats['max_requests_per_sec']['test'],
            'requests_per_sec_delta_perc': stats['max_requests_per_sec']['difference']['percentage'],
            'min_latency_99_delta_ms': stats['min_latency_99']['difference']['absolute'] * 0.001,
            'min_fail_rate': stats['min_fail_rate']['test'],
            'max_memory_rss': stats['max_memory_rss']['test'],
            'max_memory_vsz': stats['max_memory_vsz']['test'],
            'response_size_50_delta_perc': stats['response_size_50']['difference']['percentage'],
        }
        return results

    YES = '<span style="color:#09ff09;">Yes</span>'
    NO = '<span style="color:#ff0909;">No</span>'

    def get_autodeploy_footer(self, mode='default'):
        verdict = 'No info'
        if self.Context.geosuggest_autodeploy is not None:
            verdict = self.YES if self.Context.geosuggest_autodeploy else self.NO
        if mode == 'ST':
            verdict = '!!(green)YES!!' if self.Context.Geosuggest_autodeploy else '!!(red)NO!!'
            return 'Autodeploy: {verd}'.format(verd=verdict)
        return [{
            "helperName": "autodeploy",
            "content": {"<h3>Autodeploy: {0}</h3>".format(verdict): ""},
        }]

    column_measurment = "Measurement"
    column_value = "Value"
    column_min = "Min"
    column_max = "Max"
    column_autodeploy = "Autodeploy?"

    def add_measurement_min(self, rows, name, value, min_value, format_string="{}"):
        passed = value >= min_value
        rows.append(OrderedDict([
            (self.column_measurment, name),
            (self.column_value, format_string.format(value)),
            (self.column_min, "&ge;&nbsp;" + format_string.format(min_value)),
            (self.column_max, ""),
            (self.column_autodeploy, self.YES if passed else self.NO),
        ]))
        return passed

    def convert_perf_footer_to_startrek(self, data):
        rows = []
        head = '||{} ||'.format(' | '.join(data[0].keys()))
        for entry in data:
            values = [entry.get(key) for key in entry.keys()]
            row = '|| {} ||'.format(' | '.join(values))
            row = row.replace('<span style="color:#09ff09;">Yes</span>', '!!(green)YES!!')
            row = row.replace('<span style="color:#ff0909;">No</span>', '!!(red)NO!!')
            rows.append(row)
        comment = '#| ' + head + ''.join(rows) + ' |#'
        return 'Performance test results:\n{}'.format(comment)

    def add_measurement_max(self, rows, name, value, max_value, format_string="{}"):
        passed = value <= max_value
        rows.append(OrderedDict([
            (self.column_measurment, name),
            (self.column_value, format_string.format(value)),
            (self.column_min, ""),
            (self.column_max, "&le;&nbsp;" + format_string.format(max_value)),
            (self.column_autodeploy, self.YES if passed else self.NO),
        ]))
        return passed

    def get_performance_footer(self, mode='default'):
        if not self.Parameters.geosuggest_run_performance_tests:
            return [{
                "helperName": "performance",
                "content": {"<h3>Performance</h3>": "skipped"},
            }], False

        performance_summary = 'No performance test result'
        try:
            rows = []
            results = self.extract_performance_results(self.Context.subtask_performance["id"])
            passed = all([
                self.add_measurement_min(rows, "RPS", results['max_requests_per_sec'], 250, "{:.0f}"),
                # RPS: negative delta means slowdown
                self.add_measurement_min(rows, "RPS relative delta (%)", results['requests_per_sec_delta_perc'], -2., "{:.3f}"),
                # Timing: positive delta means slowdown
                self.add_measurement_max(rows, "P99 absolute delta (ms)", results['min_latency_99_delta_ms'], 5., "{:.0f}"),
                # Size: positive delta means slowdown
                self.add_measurement_max(rows, "Response size P50 relative delta (%)", results['response_size_50_delta_perc'], 5., "{:.3f}"),
                self.add_measurement_max(rows, "Fail rate", results['min_fail_rate'], 0.001),
                self.add_measurement_max(rows, "RSS", results['max_memory_rss'], 142000000),
                self.add_measurement_max(rows, "VSZ", results['max_memory_vsz'], 142000000),
            ])
            performance_summary = rows
        except Exception as e:
            performance_summary = 'Failed getting performance test results: {0}'.format(str(e))
            passed = False
        if mode == 'ST':
            return self.convert_perf_footer_to_startrek(performance_summary)
        return [{
            "helperName": "performance",
            "content": {"<h3>Performance</h3>": performance_summary},
        }], passed

    def check_suite_quality(self, row):
        actual_results = []
        expected_results = []
        autodeploy = False
        suite = row['Suite']
        parsed_link = re.match("<a.*?>(.*?)</a>", suite)
        if parsed_link is not None:
            suite = parsed_link.groups()[0]
        for condition in ACCEPTANCE_QUALITY_CONDITIONS:
            if re.match(condition['suite_regex'], suite) is not None:
                try:
                    autodeploy = True
                    if condition.get('delta_percent') is not None:
                        expected_results.append('&Delta; &gt; -{0}%'.format(condition['delta_percent']))
                        delta = int(row['Fixed']) - int(row['Broken'])
                        total_count = int(row['Total'])
                        percent = 100 * (float(delta) / float(total_count)) if total_count > 0 else 0.0
                        actual_results.append('&Delta; {:+7.2f} %'.format(percent))
                        autodeploy = autodeploy and percent > -condition['delta_percent']
                    if condition.get('max_failed') is not None:
                        expected_results.append('fails &le; {0}'.format(condition['max_failed']))
                        fails = int(row['Broken']) + int(row['Still Failed'])
                        actual_results.append('{:7}'.format(fails))
                        autodeploy = autodeploy and fails <= condition['max_failed']
                except Exception:
                    logging.exception("Failed to check suite '%s' quality", suite)
                    autodeploy = False
                break
        actual_result = ', '.join(actual_results)
        expected_result = ', '.join(expected_results)
        return actual_result, expected_result, autodeploy

    def download_content(cls, resource, filename=None):
        url = resource.http_proxy
        if filename:
            url = os.path.join(url, filename)
        try:
            response = urllib2.urlopen(url)
            return response.read()
        except urllib2.HTTPError, e:
            if e.code == 404:
                return '<a href="{0}">{1}</a> is empty or does not exist'.format(url, filename)

    def get_functional_footer(self, mode='default'):
        try:
            resource = sdk2.Resource[self.Context.FINAL_PRIEMKA_REPORT]
            content = self.download_content(resource, 'footer.json')
            footer = json.loads(content, object_pairs_hook=OrderedDict)
        except Exception:
            logging.exception("Trying old resource")
            resource = sdk2.Resource[self.Context.FINAL_DIFF_REPORT_RESOURCE]
            if not resource:
                return [{
                    "helperName": "quality_footer_error",
                    "content": "<p>No FINAL_DIFF_REPORT_RESOURCE found, can't render proper footer for quality</p>",
                }]
            footer = []
            # for compatibility with old tasks
            content = self.download_content(resource, 'autotest_diff.out.txt')
            footer = GeoSuggestTestFunctionalDiff.get_footer_by_content(content, limit=800)
        autodeploy = True
        for section in footer:
            if section.get('helperName', '?') == '':
                for table_name in section['content']:
                    table = section['content'][table_name]
                    for row in table:
                        actual_result, expected_result, suite_autodeploy = self.check_suite_quality(row)
                        row['Actual'] = actual_result
                        row['Expected'] = expected_result
                        row['Autodeploy?'] = self.YES if suite_autodeploy else self.NO
                        autodeploy = autodeploy and suite_autodeploy
        if mode == 'ST':
            content = footer[0]['content']['<h3>Quality</h3>']
            return self.convert_functional_to_startrek(content)
        return footer, autodeploy

    def convert_functional_to_startrek(self, data):
        keys = ['Suite',
                'Total',
                'Ok',
                'Fixed',
                'Broken',
                'Still Failed',
                'Actual',
                'Expected',
                'Autodeploy?']
        head = '||{} ||'.format(' | '.join(keys))
        rgx = re.compile(r'<a href="(?P<link>.*)">(?P<text>.*)</a>')
        rows = []
        for entry in data:
            values = []
            for key in keys:
                raw_value = entry.get(key, '')
                m = rgx.match(raw_value)
                if m is not None:
                    link = m.group('link')
                    text = m.group('text')
                    value = '(({link} {text}))'.format(link=link, text=text)
                    values.append(value)
                else:
                    values.append(raw_value)
            row = ' | '.join(values)
            row = row.replace('<span style="color:#09ff09;">Yes</span>', '!!(green)YES!!')
            row = row.replace('<span style="color:#ff0909;">No</span>', '!!(red)NO!!')
            rows.append('|| {} ||'.format(row))
        comment = '#| ' + head + ''.join(rows) + ' |#'
        logging.info('Functional comment:\n{comment}'.format(comment=comment))
        return 'Functional tests results:\n{}'.format(comment)

    def write_startrek_comments(self):
        try:
            autodeploy = self.get_autodeploy_footer(mode='ST')
        except Exception:
            autodeploy = ('!!(red)Failed to get autodeploy comment. '
                          'See acceptance task for details!!')
        try:
            performance = self.get_performance_footer(mode='ST')
        except Exception:
            performance = ('!!(red)Failed to get performance comment. '
                           'See acceptance task for details!!')
        try:
            functional = self.get_functional_footer(mode='ST')
        except Exception:
            functional = ('!!(red)Failed to get functional comment. '
                          'See acceptance task for details!!')
        for comment in [autodeploy, performance, functional]:
            logging.info('Trying to post %s' % comment)
            self.write_to_startrek(comment)

    @sdk2.footer()
    def footer(self):
        if self.status not in ctt.Status.Group.FINISH:
            return "<p>Results are not ready yet</p>"
        autodeploy_footer = self.get_autodeploy_footer()
        performance_footer = self.get_performance_footer()[0]
        functional_footer = self.get_functional_footer()[0]
        return autodeploy_footer + performance_footer + functional_footer
