import os

from sandbox.sandboxsdk.parameters import LastReleasedResource, SandboxStringParameter
from sandbox.sandboxsdk.process import run_process

from sandbox.projects.geosuggest import resources
from sandbox.projects.geosuggest.common import qp
from sandbox.projects.geosuggest.common.parameters import (
    AdditionalEnvParameter,
    GeoSuggestYTPoolParameter,
    GeoSuggestYTProxyParameter,
    SandboxVaultOwner,
    SandboxVaultYtTokenName
)

from sandbox.projects.common.utils import get_or_default


class GeoSuggestSearchGroupsBuilder(LastReleasedResource):
    name = 'geosuggest_search_groups_builder'
    description = 'search groups indexer'
    required = True
    resource_type = resources.GEO_SUGGEST_SEARCH_GROUPS_BUILDER


class GeobaseSnapshot(SandboxStringParameter):
    name = 'geobase_snapshot'
    description = 'geobase snapshot'
    default_value = '//home/altay/db/export/current-state/exported/company'
    required = True


class ToponymOrgDuplicates(SandboxStringParameter):
    name = 'toponym_org_duplicates'
    description = 'toponym org duplicates'
    default_value = '//home/altay/toponyms/current-state/mapping'
    required = True


class GeoSuggestBuildSearchGroups(qp.GeoSuggestQPTask):

    type = 'GEO_SUGGEST_BUILD_SEARCH_GROUPS'

    input_parameters = [
        GeoSuggestSearchGroupsBuilder,
        GeobaseSnapshot,
        ToponymOrgDuplicates,
        SandboxVaultOwner,
        SandboxVaultYtTokenName,
        GeoSuggestYTPoolParameter,
        GeoSuggestYTProxyParameter
    ]

    TIMEOUT = 10 * 3600
    execution_space = 10 * 1024  # 10 Gb
    disk_space = 10 * 1024
    cores = 1
    ram = 8192

    def get_yt_token(self):
        vault_owner = get_or_default(self.ctx, SandboxVaultOwner)
        vault_name = get_or_default(self.ctx, SandboxVaultYtTokenName)
        return self.get_vault_data(vault_owner, vault_name)

    def on_execute(self):
        AdditionalEnvParameter.ApplyToEnviron(self.ctx)

        data_builder_dir = self.sync_resource(
            get_or_default(self.ctx, GeoSuggestSearchGroupsBuilder),
        )

        org_org = get_or_default(self.ctx, GeobaseSnapshot)
        org_top = get_or_default(self.ctx, ToponymOrgDuplicates)

        output_dir = os.path.join(self.abs_path(), "output_data")
        run_process(['mkdir', output_dir])

        proc_env = os.environ.copy()
        proc_env['YT_TOKEN'] = self.get_yt_token()
        yt_pool = get_or_default(self.ctx, GeoSuggestYTPoolParameter)
        if yt_pool:
            proc_env['YT_POOL'] = yt_pool

        result_file = os.path.join(output_dir, 'search_groups')

        run_process([
            data_builder_dir,
            "--export", org_org,
            "--toponym-to-org-dups", org_top,
            "--server", get_or_default(self.ctx, GeoSuggestYTProxyParameter),
            "--result", result_file
            ], environment=proc_env)

        self.create_resource(
            'search groups',
            os.path.join(output_dir, 'search_groups'),
            resources.MAPS_GEO_SUGGEST_SEARCH_GROUPS,
        )

# check if file size less than 70 MB, it usually means something goes wrong
        file_size_mb = os.path.getsize(result_file) >> 20
        if file_size_mb < 70:
            raise RuntimeError("Searchgroups suspiciously small, make double check before release")

__Task__ = GeoSuggestBuildSearchGroups
