# -*- coding: utf-8 -*-

import re
import urllib

from sandbox import sdk2
from sandbox.projects.common.search import requester_core
from sandbox.projects.common.search import requester_mixin
from sandbox.projects.geosuggest import component
from sandbox.projects import resource_types


class Transformer(requester_core.QueryTransformer):
    def __init__(self, base_url, experiments):
        self._base_url = base_url
        self._experiments = list(experiments)

    def apply(self, query):
        if self._experiments:
            ndx = hash(query) % len(self._experiments)
            new_experiments = [self._experiments[ndx]]
        else:
            new_experiments = []
        return self._base_url + component.request.set_experiments(query, new_experiments)


def parse_list_of_ints(s):
    return [int(token) for token in s.split(',') if token]


class CommaSeparatedListOfInts(sdk2.parameters.String):
    @classmethod
    def cast(cls, value):
        if value is not None:
            parse_list_of_ints(value)
        return super(CommaSeparatedListOfInts, cls).cast(value)


class GeoSuggestCheckCrashesInExperiments(requester_mixin.RequesterTask):
    '''
    Runs geosuggest daemon and shoots it with requests setting random `&exprt=...` parameters.
    Checks that the server does not crash.
    '''

    class Requirements(component.ProductionLikeRequirements):
        pass

    class Parameters(requester_mixin.RequesterTask.Parameters):
        geosuggest = component.SingleGeosuggestParameters()

        with sdk2.parameters.String('Set of experiments') as experiments:
            experiments.values.ALL = 'all possible'
            experiments.values.FROM_DAEMON = 'ask the daemon about experiments it supports'
            experiments.values.CUSTOM = 'custom set'
        with experiments.value['CUSTOM']:
            custom_experiments = CommaSeparatedListOfInts('Custom set of experiments (comma-separated integers)')

    def on_execute(self):
        self._geosuggest = component.GeoSuggestDaemonWrapper(
            sdk2.ResourceData(self.Parameters.geosuggest.daemon).path,
            sdk2.ResourceData(self.Parameters.geosuggest.data).path,
            self.log_path('geosuggestd'),
            isolated_mode=self.Parameters.geosuggest.isolated_mode
        )

        output_responses = sdk2.ResourceData(resource_types.PLAIN_TEXT_QUERIES(
            self, 'Failed requests, {}'.format(self.Parameters.description),
            'requests.txt'
        ))

        with self._geosuggest:
            self._supported_experiments = self._get_supported_experiments(self._geosuggest)

            self._requests_failed = 0
            self._requests_total = 0
            try:
                self.save_responses(self._geosuggest.port, str(output_responses.path))
            finally:
                self.set_info('Failed request count: {} of {}'.format(self._requests_failed, self._requests_total))

        output_responses.ready()

    def _get_supported_experiments(self, geosuggest):
        response = urllib.urlopen('http://localhost:{}/suggest-dict-info'.format(geosuggest.port)).read()
        m = re.search('^Supported experiments: (.*)$', response, re.MULTILINE)
        if m is None:
            return []
        experiments = parse_list_of_ints(m.group(1))
        return experiments

    def query_transformer(self, base_url):
        self._base_url = base_url
        experiments = []

        if self.Parameters.experiments == 'ALL':
            experiments = range(1, 1024)
        elif self.Parameters.experiments == 'FROM_DAEMON':
            experiments = self._supported_experiments
        elif self.Parameters.experiments == 'CUSTOM':
            experiments = parse_list_of_ints(self.Parameters.custom_experiments)

        self.set_info('Experiments to check: {}'.format(experiments))
        return Transformer(base_url, experiments)

    def write_responses(self, out_f, r_data, r_num):
        self._requests_total += 1

        if r_data is None:
            return
        self._requests_failed += 1

        assert b'\n' not in r_data
        assert r_data.startswith(self._base_url)
        out_f.write(r_data[len(self._base_url):])
        out_f.write(b'\n')

    @property
    def stop_on_error(self):
        return not self._geosuggest.running

    def post_process_response(self, r_num, req, r_status, r_data):
        if r_status:
            # successful response
            return None
        return req
