# -*- coding: utf-8 -*-

import json
import logging
import os
import jinja2
from sandbox import sdk2
from sandbox.projects.common import file_utils
from sandbox.sdk2.helpers import ProcessLog, subprocess


ARCADIA_SCRIPT_PATH = "arcadia:/arc/trunk/arcadia/quality/trailer/suggest/services/maps/tools/personalization/monitoring/check_personalization_health.py"
VERDICT_HAS_LOST_CLICKS = "Has lost clicks"
VERDICT_NO_CLICKS_LOST = "No clicks lost"


class GeoSuggestCheckPersonalizationHealth(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1024  # 1 Gb
        disk_space = 1024  # 1Gb

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600 * 3 + 600

        clicks_per_uuid = sdk2.parameters.Integer("Clicks per uuid", default=10, description="Amount of clicks sent for each uuid")
        data_update_timeout = sdk2.parameters.Integer("Personal data update timeout", default=3600 * 3, description="Max waiting time in seconds for personal data update after click imitation")

        upload_report_to_stat = sdk2.parameters.Bool("Upload report to Stat", default=True, description="If set, the report about click timings and losses will be uploaded into stat.yandex-team.ru")
        with upload_report_to_stat.value[True]:
            stat_token_vault_item_name = sdk2.parameters.String("Stat token vault item name", default="stat_token", description="Vault item name for stat.yandex-team.ru oauth token")
            stat_token_vault_item_owner = sdk2.parameters.String("Stat token vault item owner", default="GEO_SUGGEST", description="Vault item owner for stat.yandex-team.ru oauth token")

        with sdk2.parameters.Output:
            losses_verdict = sdk2.parameters.String(
                "Losses verdict",
                description="Can be '{}' or '{}'. You can search tasks by verdict, entering these strings in Task Hint field".format(VERDICT_HAS_LOST_CLICKS, VERDICT_NO_CLICKS_LOST),
                hint=True
            )

    def on_execute(self):
        logging.info("Acquiring health check script from %s...", ARCADIA_SCRIPT_PATH)
        local_script_path = os.path.basename(ARCADIA_SCRIPT_PATH)
        sdk2.vcs.svn.Arcadia.export(ARCADIA_SCRIPT_PATH, local_script_path)

        report_path = "report.json"
        cmd_line_args = [
            "python", local_script_path,
            "--clicks_per_uuid", str(self.Parameters.clicks_per_uuid),
            "--timeout", str(self.Parameters.data_update_timeout),
            "--output_file", report_path,
        ]

        if self.Parameters.upload_report_to_stat:
            logging.info("Acquiring stat token from vault...")
            stat_token = sdk2.Vault.data(self.Parameters.stat_token_vault_item_owner, self.Parameters.stat_token_vault_item_name)
            cmd_line_args += [
                "--upload_to_stat",
                "--stat_token", stat_token,
            ]

        logging.info("Running health check script...")
        with ProcessLog(self, logger="check-health") as pl:
            subprocess.check_call(cmd_line_args, stdout=pl.stdout, stderr=pl.stdout)

        logging.info("Writing report content into task context...")
        report_content = json.loads(file_utils.read_file(report_path))
        health_check_summary = {}
        has_lost_clicks = False
        for click_type in report_content:
            health_check_summary[click_type] = report_content[click_type]["summary"]
            lost_clicks = report_content[click_type]["details"].get("not_found_clicks", [])
            if lost_clicks:
                health_check_summary[click_type]["lost_queries"] = [click["query_text"] for click in lost_clicks]
                has_lost_clicks = True
        self.Context.health_check_summary = json.dumps(health_check_summary)
        self.Context.has_lost_clicks = has_lost_clicks
        self.Parameters.losses_verdict = VERDICT_HAS_LOST_CLICKS if has_lost_clicks else VERDICT_NO_CLICKS_LOST

        logging.info("Finished")

    @sdk2.footer()
    def results(self):
        if not self.Context.health_check_summary:
            return "Task execution in progress..."
        task_source_dir = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(task_source_dir), extensions=["jinja2.ext.do"])
        return env.get_template("footer.html.jinja2").render(health_check_summary=json.loads(self.Context.health_check_summary))
