import logging
import urllib

from sandbox import sdk2
from sandbox.common.errors import TaskError, VaultNotFound
from sandbox.projects.common import dolbilka2
from sandbox.projects import resource_types as rt
from sandbox.projects.geosuggest import binary_task
from sandbox.projects.geosuggest import resources as gsr
from sandbox.projects.geosuggest.component.request import set_experiments

QUERY = '''
USE hahn;

PRAGMA yson.DisableStrict;

DECLARE $cnt AS int;
DECLARE $min_response_time AS int;
DECLARE $with_zero_suggest AS bool;

$isGood = ($fullRequest) -> {
    RETURN ($fullRequest LIKE '/suggest-geo%') OR IF ($with_zero_suggest, $fullRequest LIKE '/zero-suggest%', False)
};

$table_paths = (
    SELECT AGGREGATE_LIST(Path)
    FROM (
        SELECT Path FROM
        FOLDER("//logs/suggest-maps-reqans-log/1d")
        WHERE Type = "table"
        ORDER BY Path DESC LIMIT 3
    )
);

SELECT
    fullRequest,
    Yson::LookupString(userId, 'passportUid'),
    Yson::LookupString(userId, 'yandexLogin'),
    Yson::LookupString(userId, 'yandexUid'),
    Yson::LookupString(userId, 'uuid')
FROM EACH($table_paths)
WHERE responseTime > $min_response_time AND $isGood(fullRequest)
ORDER BY RANDOM(TableRow()) LIMIT $cnt;
'''

TEXT_REQUESTS_FILENAME = 'requests.txt'
BINARY_PLAN_FILENAME = 'geosuggest.plan'


def _to_bin_str(value):
    if value is None:
        return None
    if isinstance(value, str):
        return value
    elif isinstance(value, unicode):
        return value.encode('utf-8')
    else:
        logging.error('got a non-string result: %r', value)
        raise TaskError('Invalid data received from YQL')


class GeoSuggestCollectRequests(binary_task.LastBinaryTaskRelease, sdk2.Task):
    '''
    Collects geosuggest requests from reqans logs on YT.
    '''

    class Requirements(sdk2.Requirements):
        cpu = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        count = sdk2.parameters.Integer('Number of requests to collect',
                                        required=True,
                                        default=1000)
        min_response_time = sdk2.parameters.Integer('Min response time (ms)',
                                                    required=False,
                                                    default=0)
        with_zero_suggest = sdk2.parameters.Bool('Collect zero-suggest requests as well',
                                                 required=False,
                                                 default=False)
        append_personal_debug_params = sdk2.parameters.Bool('Append personal &debug_... parameters',
                                                            required=False,
                                                            default=False)
        remove_exprt_params = sdk2.parameters.Bool('Remove experimental &exprt... parameters',
                                                   required=False,
                                                   default=True)
        ext_params = binary_task.binary_release_parameters(stable=True)

    def _collect_requests(self, output_name):
        from yql.api.v1.client import YqlClient
        from yql.client.parameter_value_builder import YqlParameterValueBuilder as ValueBuilder

        try:
            token = sdk2.Vault.data(self.owner, 'yql_token')
        except VaultNotFound:
            token = sdk2.Vault.data(self.owner, 'YQL_TOKEN')

        client = YqlClient(token=token)

        parameters = {
            '$cnt': ValueBuilder.make_int32(self.Parameters.count),
            '$min_response_time': ValueBuilder.make_int32(self.Parameters.min_response_time),
            '$with_zero_suggest': ValueBuilder.make_bool(self.Parameters.with_zero_suggest)
        }
        request = client.query(query=QUERY, syntax_version=1)
        request.run(parameters=ValueBuilder.build_json_map(parameters))

        results = request.get_results()

        if not request.is_success:
            for error in request.errors:
                logging.error('YQL error: %s', error)
            raise TaskError('Error while running YQL query, see logs for details')

        with open(output_name, 'wb') as fd:
            for table in results:
                table.fetch_full_data()
                for row in table.rows:
                    full_request, passport_uid, login, yandexuid, uuid = map(_to_bin_str, row)

                    if self.Parameters.remove_exprt_params:
                        full_request = set_experiments(full_request, [])

                    extra = []
                    if self.Parameters.append_personal_debug_params:
                        if passport_uid:
                            extra.append(('debug_uid', passport_uid))
                        if login:
                            extra.append(('debug_login', login))
                        if yandexuid:
                            extra.append(('debug_yandexuid', yandexuid))
                        if uuid:
                            extra.append(('debug_uuid', uuid))

                    if extra:
                        full_request += '&' + urllib.urlencode(extra)

                    fd.write(full_request)
                    fd.write(b'\n')

    def on_execute(self):
        self._collect_requests(TEXT_REQUESTS_FILENAME)
        resource = rt.PLAIN_TEXT_QUERIES(self, self.Parameters.description, TEXT_REQUESTS_FILENAME)
        sdk2.ResourceData(resource).ready()

        planner = dolbilka2.DolbilkaPlanner2()
        planner.create_plan(TEXT_REQUESTS_FILENAME, BINARY_PLAN_FILENAME, host='localhost', port=9960)
        resource = gsr.GEO_SUGGEST_WEBDAEMON_PLAN(self, self.Parameters.description, BINARY_PLAN_FILENAME)
        sdk2.ResourceData(resource).ready()
