# -*- coding: utf-8 -*-

from sandbox import common
from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.projects.common import link_builder
from sandbox.projects.common.differ.json_differ import JsonDiffer
from sandbox.projects.common.differ.printers import PrinterToHtml
from sandbox.projects.geosuggest import resources


def _generate_pairs(requests, responses1, responses2):
    while True:
        request = requests.readline()
        response1 = responses1.readline()
        response2 = responses2.readline()
        if response1 == '' and response2 == '':
            break
        if request == '' or response1 == '' or response2 == '':
            raise common.errors.TaskFailure('Numbers of requests do not match')
        yield (response1, response2, request)


class GeoSuggestCompareResponses(sdk2.Task):
    '''
    Compares geosuggest responses stored as JSON lines.
    '''

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        requests = sdk2.parameters.Resource('Requests', resource_type=resource_types.PLAIN_TEXT_QUERIES, required=True)
        responses1 = sdk2.parameters.Resource('Respones #1', resource_type=resources.GeoSuggestJsonResponses, required=True)
        responses2 = sdk2.parameters.Resource('Respones #2', resource_type=resources.GeoSuggestJsonResponses, required=True)

        with sdk2.parameters.Output:
            has_diff = sdk2.parameters.Bool('Has diff', required=True, default=True)

    def on_execute(self):
        if self._check_md5():
            # Fast pass: check hashes only
            self.Parameters.has_diff = False
            return

        diff_dir = sdk2.path.Path('diff')
        diff_dir.mkdir()

        diffs_found = self._compare(diff_dir)
        self.set_info('Differences found: {}'.format(diffs_found))

        if diffs_found > 0:
            self.Parameters.has_diff = True
            self._publish_resource(diff_dir)
        else:
            self.Parameters.has_diff = False

    def _check_md5(self):
        hash1 = self.Parameters.responses1.md5
        hash2 = self.Parameters.responses2.md5
        if hash1 and hash2 and hash1 == hash2:
            self.set_info('Responses are equal: same MD5 digest {}'.format(hash1))
            return True

    def _compare(self, diff_dir):
        printer = PrinterToHtml(
            str(diff_dir),
            write_compact_diff=False,
            pair_head_template='response {obj_index}',
            obj_head_template='query'
        )
        differ = JsonDiffer(printer)

        with sdk2.ResourceData(self.Parameters.requests).path.open('rb') as requests_file:
            with sdk2.ResourceData(self.Parameters.responses1).path.open('rb') as responses1_file:
                with sdk2.ResourceData(self.Parameters.responses2).path.open('rb') as responses2_file:
                    differ.compare_pairs(_generate_pairs(requests_file, responses1_file, responses2_file), process_parallel=False)

        return differ.get_diff_count()

    def _publish_resource(self, diff_dir):
        res = resource_types.BASESEARCH_RESPONSES_COMPARE_RESULT(
            self, 'Compare result, {}'.format(self.Parameters.description), diff_dir
        )
        sdk2.ResourceData(res).ready()
        self.set_info('Diff: {}'.format(link_builder.resource_link(res.id)), do_escape=False)
