# -*- coding: utf-8 -*-

import json

from sandbox import sdk2
from sandbox.projects.common.search import requester_core
from sandbox.projects.common.search import requester_mixin
from sandbox.projects.common.differ.json_differ import JsonDiffer
from sandbox.projects.geosuggest import component
from sandbox.projects.geosuggest import resources
from sandbox.projects.geosuggest import binary_task


class Transformer(requester_core.QueryTransformer):
    def __init__(self, base_url, override_parameters):
        self._base_url = base_url
        self._override_parameters = override_parameters

    def apply(self, query):
        return self._base_url + component.request.stabilize(query, self._override_parameters)


class GeoSuggestGetResponses(binary_task.LastBinaryTaskRelease, requester_mixin.RequesterTask):
    '''
    Runs geosuggest daemon and stores the responses as JSON.
    '''

    class Requirements(component.ProductionLikeRequirements):
        pass

    class Parameters(requester_mixin.RequesterTask.Parameters):
        geosuggest = component.SingleGeosuggestParameters()
        ext_params = binary_task.binary_release_parameters(stable=True)
        override_parameters = sdk2.parameters.String('Override query string parameters', default='', description='In the format of key1=value1&key2=value2&...')

    def on_execute(self):
        geosuggest = component.GeoSuggestDaemonWrapper(
            sdk2.ResourceData(self.Parameters.geosuggest.daemon).path,
            sdk2.ResourceData(self.Parameters.geosuggest.data).path,
            self.log_path('geosuggestd'),
            isolated_mode=self.Parameters.geosuggest.isolated_mode,
            isolated_enable_personalization=self.Parameters.geosuggest.isolated_enable_personalization
        )

        output_responses = sdk2.ResourceData(resources.GeoSuggestJsonResponses(
            self, 'Geosuggest responses, {}'.format(self.Parameters.description),
            'responses.jsonl'
        ))

        with geosuggest:
            self.save_responses(
                geosuggest.port,
                str(output_responses.path),
            )
        output_responses.ready()

    def post_process_response(self, r_num, req, r_status, r_data):
        from sandbox.projects.geosuggest.response.parse import response_as_object
        if r_status:
            try:
                obj = response_as_object(r_data)
            except Exception as e:
                obj = {'__type__': type(e).__name__, '__error__': str(e)}
        else:
            obj = {'__error__': r_data}

        return json.dumps(obj, ensure_ascii=False).encode('utf-8')

    def query_transformer(self, base_url):
        return Transformer(base_url, self.Parameters.override_parameters)

    def write_responses(self, out_f, r_data, r_num):
        n_repeats = self.Parameters.recheck_n + 1
        if r_num % n_repeats != 0:
            return

        assert b'\n' not in r_data
        out_f.write(r_data)
        out_f.write(b'\n')

    def create_differ(self, output_printer):
        return JsonDiffer(output_printer)

    @property
    def stop_on_error(self):
        return False
