import json
import os
import shutil
from distutils import dir_util
import subprocess
import logging

from sandbox import sandboxsdk
from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.projects.geosuggest import component
from sandbox.projects.geosuggest import resources
from sandbox.projects.common.nanny import nanny
from sandbox.projects.geosuggest.common import qp
from sandbox.projects.geosuggest.common.parameters import (GeoSuggestDataBuilderParameter, GeoSuggestSandboxBinParameter, AdditionalEnvParameter)


class GeoSuggestDataBeforePostprocessing(sandboxsdk.parameters.ResourceSelector):
    name = component.GEO_SUGGEST_DATA_CTX_KEY
    required = True
    description = 'Geo suggest data before postprocessing'
    resource_type = resources.GEO_SUGGEST_DATA_BEFORE_POSTPROCESSING
    group = component._GEO_SUGGEST_PARAMETERS_GROUP


class GeoSuggestPostprocessing(nanny.ReleaseToNannyTask, component.GeoSuggestDaemonTester, qp.GeoSuggestQPTask):

    type = 'GEO_SUGGEST_POSTPROCESSING'
    input_parameters = [
        component.GeoSuggestDaemonParameter,
        GeoSuggestSandboxBinParameter,
        GeoSuggestDataBeforePostprocessing,
        component.GeoSuggestConfigParameter,
        component.GeoSuggestStartTimeoutParameter,
        component.GeoSuggestShutdownTimeoutParameter,
        GeoSuggestDataBuilderParameter,
        AdditionalEnvParameter,
    ]
    execution_space = 100 * 1024  # 80 Gb
    required_ram = 120 * 1024  # 120 Gb

    TIMEOUT = 6 * 3600

    def initCtx(self):
        self.ctx['kill_timeout'] = 6 * 60 * 60  # 6 hours

    def on_execute(self):
        AdditionalEnvParameter.ApplyToEnviron(self.ctx)

        # Make a read-write local copy of a shard
        pack_dir = os.path.join(self.abs_path(), 'pack')
        data_dir = os.path.join(pack_dir, 'data/')
        dir_util.copy_tree(self.sync_resource(self.ctx[GeoSuggestDataBeforePostprocessing.name]), pack_dir)

        prepare_data_bin = os.path.join(self.sync_resource(self.ctx[GeoSuggestSandboxBinParameter.name]), 'suggest_prepare_data')
        geobase_path = os.path.join(data_dir, 'geobase/geodata5.bin')
        output_dir = os.path.join(data_dir, 'toponyms_bin')
        toponyms_prepare_cmd = [prepare_data_bin,
                                'toponyms',
                                '-d', data_dir,
                                '-g', geobase_path,
                                '-o', output_dir]
        log_file_path = os.path.join(get_logs_folder(), 'toponyms_prepare.log')
        with open(log_file_path, 'w') as log_file:
            logging.info('Running: %s' % toponyms_prepare_cmd)
            subprocess.check_call(toponyms_prepare_cmd,
                                  stdout=log_file,
                                  stderr=subprocess.STDOUT)
        toponyms_xml_dir = os.path.join(data_dir, 'toponyms')
        shutil.rmtree(toponyms_xml_dir)

        description_data = {'MinShard': 0,
                            'MaxShard': 4294967295,
                            'ObjsLimit': 0,
                            'GeoObjectsShiftStart': 0,
                            'SelectedGeoID': 0}
        description_file_path = os.path.join(output_dir, 'description')
        json.dump(description_data, open(description_file_path, 'w'))

        with self.start_daemon(custom_data_dir=pack_dir):
            pass

        resource = self.create_resource(
            self.descr,
            pack_dir,
            resources.GEO_SUGGEST_DATA,
        )
        self.mark_resource_ready(resource.id)

        self.ctx["nanny_release_info"] = json.dumps(self.get_nanny_release_info({"release_status": "prestable"}))

    def on_release(self, additional_parameters):
        self.ctx["nanny_release_info"] = json.dumps(self.get_nanny_release_info(additional_parameters))

        qp.GeoSuggestQPTask.on_release(self, additional_parameters)
        if qp.GeoSuggestQPTask.qp_passed:
            nanny.ReleaseToNannyTask.on_release(self, additional_parameters)

    @classmethod
    def on_qp_tests(cls, task, rt, prev, curr, env):
        if rt == resources.GEO_SUGGEST_DATA:
            pass


__Task__ = GeoSuggestPostprocessing
