import os
import urllib2
import json
import collections
from distutils import dir_util
from distutils import file_util

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.environments import VirtualEnvironment
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.paths import get_logs_folder

from sandbox.projects.geosuggest import resources
from sandbox.projects.geosuggest.common.parameters import (
    GeoSuggestDataBuilderParameter
)
from sandbox.projects.common.utils import get_or_default

import sandbox.projects.geosuggest.component as geo_suggest_c


class GeoSuggestTestFunctionalDiff(geo_suggest_c.GeoSuggestDaemonTester):

    type = 'GEO_SUGGEST_TEST_FUNCTIONAL_DIFF'

    class GeoSuggestAutotestReportOld(parameters.ResourceSelector):
        name = 'autotest_report_left_id'
        description = 'GeoSuggestTestFunctional logs old'
        resource_type = resources.GEO_SUGGEST_TEST_FUNCTIONAL_REPORT

    class GeoSuggestAutotestReportNew(parameters.ResourceSelector):
        name = 'autotest_report_right_id'
        description = 'GeoSuggestTestFunctional logs new'
        resource_type = resources.GEO_SUGGEST_TEST_FUNCTIONAL_REPORT

    class GeoSuggestTestFunctionalFailOnDiff(parameters.SandboxBoolParameter):
        name = 'geosuggest_test_functional_fail_on_diff'
        description = 'Fail if diff detected'
        default_value = False

    input_parameters = [
        GeoSuggestDataBuilderParameter,
        GeoSuggestAutotestReportOld,
        GeoSuggestAutotestReportNew,
        GeoSuggestTestFunctionalFailOnDiff,
    ]

    def initCtx(self):
        self.ctx['kill_timeout'] = 5 * 3600

    def prepare_footer(self, scripts_path, diff_path, export_dir):
        with VirtualEnvironment(use_system=True) as venv:
            venv.pip('ijson')
            venv.pip('lxml')
            run_process(
                [
                    venv.executable,
                    os.path.join(scripts_path, 'tests', 'autotest_generate_htmls.py'),
                    '-i', diff_path,
                    '-o', export_dir,
                ],
                wait=True,
                log_prefix='autotest_diff',
            )
        has_diff = False
        with open(os.path.join(export_dir, 'autotest_has_diff'), 'r') as f:
            s = f.readlines()
            if len(s) == 1 and s[0].strip() == 'True':
                has_diff = True
            elif s != '':
                has_diff = s
        return has_diff

    def on_execute(self):
        data_builder_dir = self.sync_resource(
            get_or_default(self.ctx, GeoSuggestDataBuilderParameter),
        )
        data_builder_tests_dir = os.path.join(data_builder_dir, 'tests')
        autotest_diff_path = os.path.join(data_builder_tests_dir, 'autotest_diff.py')

        logs_old_dir = self.sync_resource(
            get_or_default(self.ctx, self.GeoSuggestAutotestReportOld),
        )
        report_old_path = os.path.join(logs_old_dir, 'autotest_report.json')

        logs_new_dir = self.sync_resource(
            get_or_default(self.ctx, self.GeoSuggestAutotestReportNew),
        )
        report_new_path = os.path.join(logs_new_dir, 'autotest_report.json')

        with VirtualEnvironment(use_system=True) as venv:
            venv.pip('lxml')
            autotest_diff_cmd = [
                venv.executable,
                autotest_diff_path,
                "--prev={0}".format(report_old_path),
                "--new={0}".format(report_new_path),
                "--format=json",
            ]
            run_process(
                autotest_diff_cmd,
                wait=True,
                log_prefix='autotest_diff',
            )

        log_dir = get_logs_folder()
        export_dir = os.path.join(self.abs_path(), 'test_functional_diff_report')
        dir_util.mkpath(export_dir)
        export_diff_path = os.path.join(export_dir, 'autotest_diff.out.txt')
        file_util.copy_file(os.path.join(log_dir, 'autotest_diff.out.txt'), export_diff_path)
        has_diff = self.prepare_footer(data_builder_dir, export_diff_path, export_dir)
        self.ctx['has_diff'] = has_diff
        resource = self.create_resource(
            'Geosuggest quality test diff report',
            export_dir,
            resources.GEO_SUGGEST_TEST_FUNCTIONAL_DIFF_REPORT,
        )
        self.mark_resource_ready(resource.id)

        if get_or_default(self.ctx, self.GeoSuggestTestFunctionalFailOnDiff):
            if has_diff:
                raise SandboxTaskFailureError('Diff detected')

    @classmethod
    def download_content(cls, resource, filename=None):
        url = resource.proxy_url
        if filename:
            url = os.path.join(url, filename)
        try:
            response = urllib2.urlopen(url)
            return response.read()
        except urllib2.HTTPError, e:
            if e.code == 404:
                return '<a href="{0}">{1}</a> is empty or does not exist'.format(url, filename)

    @classmethod
    def footer_get_row(cls, suite, link, names, summary, total=None):
        result = collections.OrderedDict([("Suite", suite if link == "" else "<a href=\"{0}\">{1}</a>".format(link, suite))])
        for field in names:
            result[names[field]] = str(summary[field])
            if total:
                total[field] += int(summary[field])
        return result

    @classmethod
    def footer_get_table(cls, report):
        header = "<h3>Quality</h3>"
        result = {header: []}
        names = collections.OrderedDict([("total", "Total"), ("ok", "Ok"), ("fixed", "Fixed"), ("broken", "Broken"), ("still_failed", "Still Failed")])
        total = {"total": 0, "ok": 0, "fixed": 0, "broken": 0, "still_failed": 0}
        for suite in report["tests"]:
            link = report["tests"][suite].get("link", "")
            result[header].append(cls.footer_get_row(suite.encode('utf-8'), link, names, report["tests"][suite]["summary"], total))
        result[header].append(cls.footer_get_row("Total", '', names, total))
        return result

    @classmethod
    def footer_get_diff(cls, report, limit=None):
        green = '#C5FFC4'
        yellow = '#FDFEBC'
        red = '#FFC5CE'
        fixed_begin = "<pre style='background-color:{green}'>".format(green=green)
        fixed_end = "</pre>"
        broken_begin = "<pre style='background-color:{red}'>".format(red=red)
        broken_end = "</pre>"
        still_failed_begin = "<pre style='background-color:{yellow}'>".format(yellow=yellow)
        still_failed_end = "</pre>"
        content = ""

        for suite in report["tests"]:
            content += "Tests for " + suite.encode('utf-8') + "<br/>"
            count = 0

            for test in report["tests"][suite]["list"]:
                template = ''
                begin = ""
                header = ""
                details = ""
                end = ""
                need_details = False

                if test["res"] == "ok":
                    template = ''

                elif test["res"] == "fixed":
                    template = '{begin}{header}<br/>{details}{end}'
                    need_details = True
                    begin = fixed_begin
                    end = fixed_end
                    header = 'FIXED test {number} {message}'.format(number=test["num"], message=test["new"]["message"].encode('utf-8'))
                    count = count + 1

                elif test["res"] == "broken":
                    template = '{begin}{header}<br/>{details}{end}'
                    need_details = True
                    begin = broken_begin
                    end = broken_end
                    header = 'BROKEN {number} {message}'.format(number=test["num"], message=test["new"]["message"].encode('utf-8'))
                    count = count + 1

                elif test["res"] == "still_failed":
                    template = '{begin}{header}<br/>{details}{end}'
                    begin = still_failed_begin
                    end = still_failed_end
                    header = 'STILL FAILED {0}'.format(test["new"]["message"].encode('utf-8'))
                    count = count + 1

                if need_details and "response_items" in test["new"]:

                    clicked_position = test["new"].get("position")

                    for pos, s in enumerate(test["new"]["response_items"]):
                        details += '\t[{0:^3}][{1}][{2:^16}][ {3:<60}] {4}<br/>'.format(
                            str(pos + 1),
                            'x' if pos + 1 == clicked_position else ' ',
                            s.get("type", "").encode('utf-8'),
                            s.get("title", "").encode('utf-8'),
                            s.get("text", "").encode('utf-8')
                        )

                    count = count + 1

                if test["res"] != "still_failed":
                    content += template.format(begin=begin.encode('utf-8'), end=end.encode('utf-8'), header=header, details=details)

                if limit:
                    if count >= limit:
                        content += "... showed only first {0} differences</br>".format(limit)
                        break

            content += "<br/><br/><br/><br/>"

        return content

    @classmethod
    def get_footer_by_content(cls, content, limit=None):
        report = json.loads(content)
        return [
            {
                "helperName": "",
                "content": cls.footer_get_table(report),
            },
            {
                "helperName": "diff",
                "content": cls.footer_get_diff(report, limit),
            },
        ]

    @classmethod
    def get_footer_by_resource(cls, resource, limit=None):
        content = cls.download_content(resource, "autotest_diff.out.txt")
        return cls.get_footer_by_content(content, limit)

    @property
    def footer(self):
        if not self.is_finished():
            return "<p>Results are not ready yet</p>"

        task_logs_resources = self.list_resources("TASK_LOGS")
        if not task_logs_resources:
            return "<p>No TASK_LOGS resource, can't render proper footer</p>"
        return self.get_footer_by_resource(task_logs_resources[0])


__Task__ = GeoSuggestTestFunctionalDiff
