import time

from sandbox.projects.geosuggest import component
from sandbox.projects.common.search import performance
from sandbox.projects.common import profiling


def get_process_memory(pid, executable_name):
    import psutil
    p = psutil.Process(pid)
    if p.name == executable_name:
        return p.get_memory_info()
    for child in p.get_children(recursive=True):
        if child.name == executable_name:
            return child.get_memory_info()


class GeoSuggestTestPerformance(
    performance.OldShootingTask,
    profiling.ProfilingTask,
    component.GeoSuggestDaemonTester,
):

    """
    Geo suggest daemon load testing.

    Here is a quick guide on how to use this task.

    #. Build daemon with `YA_MAKE`_ task:

        - set **Targets** to *quality/trailer/suggest/services/maps/SuggestFcgiD*
        - set **Build artifacts** to *quality/trailer/suggest/services/maps/SuggestFcgiD/geosuggestd*
        - set **Result is a single file** to *True*
        - set **Result resource type** to `GEO_SUGGEST_WEBDAEMON`_

    #. Build daemon data with `YA_MAKE`_ task:

        - set **Targets** to *quality/trailer/suggest/services/maps/data/data-package*
        - set **Build artifacts** to *quality/trailer/suggest/services/maps/data/data-package/quality/trailer/suggest/services/maps/data/data*
        - set **Result resource type** to `GEO_SUGGEST_DATA`_

    #. Prepare plan for `dolbilka`_:

        - you need a file which content looks like:

            ::

                ...
                /suggest-geo-mobile?callback=jsoncb&full...
                /suggest-geo?callback=jQuery11110940052327...
                /suggest-geo-mobile?callback=jsoncb&fullpath=1&highl...
                ...

        - let's say file name is ``cgi.log``, then you should run:

            ::

                d-planner -t plain -h 'localhost' -p 36804 -l cgi.log -o cgi.plan

        - and upload ``cgi.plan`` to Sandbox as `GEO_SUGGEST_WEBDAEMON_PLAN`_ resource

            ::

                ya upload -T GEO_SUGGEST_WEBDAEMON_PLAN cgi.plan

    #. Set parameters for the task and hit **Execute**.

    .. _GEO_SUGGEST_WEBDAEMON: https://sandbox.yandex-team.ru/docs/html/tasks/projects.html#projects.resource_types.GEO_SUGGEST_WEBDAEMON
    .. _GEO_SUGGEST_DATA: https://sandbox.yandex-team.ru/docs/html/tasks/projects.html#projects.resource_types.GEO_SUGGEST_DATA
    .. _GEO_SUGGEST_WEBDAEMON_PLAN: https://sandbox.yandex-team.ru/docs/html/tasks/projects.html#projects.resource_types.GEO_SUGGEST_WEBDAEMON_PLAN
    .. _YA_MAKE: https://sandbox.yandex-team.ru/docs/html/tasks/projects.common.build.html#projects.common.build.YaMake.YaMakeTask
    .. _dolbilka: https://wiki.yandex-team.ru/jandekspoisk/sepe/dolbilka/
    """

    type = 'GEO_SUGGEST_TEST_PERFORMANCE'
    client_tags = component.LOAD_TESTING_CLIENT_TAGS

    input_parameters = (
        component.GeoSuggestDaemonTester.input_parameters +
        (component.GeoSuggestDolbilkaPlanParameter,) +
        performance.OldShootingTask.shoot_input_parameters +
        profiling.ProfilingTask.input_parameters
    )

    def initCtx(self):
        self.ctx['fail_on_any_error'] = True

        self.ctx['enable_lunapark'] = True
        self.ctx['tank_task_name'] = 'QREG-188'

    def on_execute(self):
        geosuggestd = self.start_daemon()

        self._init_virtualenv()
        self._profiling_init(geosuggestd, self.__get_perf_data_path())

        begin_starting_up = time.time()
        with geosuggestd:
            end_starting_up = time.time()
            self.ctx.setdefault('startup_time', end_starting_up - begin_starting_up)

            self.set_info(component.log.get_startup_time_stats(geosuggestd.daemon_log))
            self.set_info(component.log.get_startup_memory_stats(geosuggestd.daemon_log))

            self._old_shoot(
                geosuggestd,
                self.ctx[component.GeoSuggestDolbilkaPlanParameter.name],
            )
        self._profiling_report(geosuggestd, self.__get_perf_data_path())

    def __get_perf_data_path(self):
        return self.abs_path('perf.data')

    @property
    def footer(self):
        footers = []

        performance_footer = performance.OldShootingTask.footer.fget(self)
        footers.append({
            'helperName': '',
            'content': performance_footer
        })

        profiling_footer = profiling.ProfilingTask.profiling_footer(self)
        if profiling_footer:
            footers.append({
                'helperName': '',
                'content': profiling_footer
            })

        return footers

    def _save_memory_usage(self, target):
        info = get_process_memory(target.process.pid, 'geosuggestd')
        if info is None:
            return

        self.ctx.setdefault(performance._MEMORY_RSS_KEY, []).append(info.rss // 1024)
        self.ctx.setdefault(performance._MEMORY_VSZ_KEY, []).append(info.vms // 1024)


__Task__ = GeoSuggestTestPerformance
