# -*- coding: utf-8 -*-

import jinja2
from contextlib import contextmanager

from sandbox import common
from sandbox import sdk2
from sandbox.projects.common import dolbilka2 as dolbilka
from sandbox.projects.common.search.performance import delta_percent
from sandbox.projects.geosuggest import component
from sandbox.projects.geosuggest import resources
from sandbox.projects.tank import executor2 as tank_executor
from sandbox.projects.tank.load_resources import resources as tank_resources
from sandbox.sandboxsdk import process


@contextmanager
def started(component):
    '''
    Context manager that only starts a component, does not wait until it is ready.
    '''
    try:
        component.start()
        yield
    finally:
        component.stop()


def mean(numbers):
    return float(sum(numbers)) / max(len(numbers), 1)


def get_resource_path_if_exists(resource):
    if resource:
        return sdk2.ResourceData(resource).path
    return None


class GeoSuggestTestPerformanceParallel(sdk2.Task):
    '''
    Runs two geosuggests on the same machine and compares their performance.
    '''

    class Requirements(sdk2.Task.Requirements):
        disk_space = component.EXECUTION_SPACE * 2
        ram = component.REQUIRED_RAM * 2
        client_tags = component.LOAD_TESTING_CLIENT_TAGS

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 5 * 3600
        isolated_mode = sdk2.parameters.Bool('Isolated mode', description='Do not use external services', default=False)

        with sdk2.parameters.Group('Geosuggest: reference') as reference_params:
            reference_daemon = sdk2.parameters.Resource(
                'Daemon',
                resource_type=resources.GEO_SUGGEST_WEBDAEMON,
                required=True)
            reference_config = sdk2.parameters.Resource(
                'Daemon config',
                resource_type=resources.GEO_SUGGEST_WEBDAEMON_CONFIG,
                required=False)
            reference_data = sdk2.parameters.Resource(
                'Data',
                resource_type=resources.GEO_SUGGEST_DATA,
                required=True)
            reference_log_level = sdk2.parameters.String(
                'Logging level',
                default=component.GeoSuggestDaemonWrapper.LOG_LEVEL_ERROR,
                choices=component.GeoSuggestDaemonWrapper.LOG_LEVEL_CHOICES,
                required=False)
            reference_aux_formula = sdk2.parameters.Resource(
                'Geo suggest aux formula resource id (experimental)',
                resource_type=resources.GEO_SUGGEST_MATRIXNET_MODEL)
            reference_aux_formulas_fml_ids = sdk2.parameters.List(
                'Geo suggest aux formulas fml-ids list (experimental)',
                default=None,
                required=False,
                value_type=sdk2.parameters.String)
            reference_aux_experiments = sdk2.parameters.List(
                'Aux experiments list (experimental)',
                default=None,
                required=False,
                value_type=sdk2.parameters.String)
            reference_custom_flags = sdk2.parameters.Dict(
                'Custom flags list (experimental)',
                default=None,
                required=False,
                value_type=sdk2.parameters.String)
            reference_custom_cmdl = sdk2.parameters.String(
                'Custom command line',
                default=None,
                required=False)

        with sdk2.parameters.Group('Geosuggest: test') as test_params:
            test_daemon = sdk2.parameters.Resource(
                'Daemon',
                resource_type=resources.GEO_SUGGEST_WEBDAEMON,
                required=True)
            test_config = sdk2.parameters.Resource(
                'Daemon config',
                resource_type=resources.GEO_SUGGEST_WEBDAEMON_CONFIG,
                required=False)
            test_data = sdk2.parameters.Resource(
                'Data',
                resource_type=resources.GEO_SUGGEST_DATA,
                required=True)
            test_log_level = sdk2.parameters.String(
                'Logging level',
                default=component.GeoSuggestDaemonWrapper.LOG_LEVEL_ERROR,
                choices=component.GeoSuggestDaemonWrapper.LOG_LEVEL_CHOICES,
                required=False)
            test_aux_formula = sdk2.parameters.Resource(
                'Geo suggest aux formula resource id (experimental)',
                resource_type=resources.GEO_SUGGEST_MATRIXNET_MODEL)
            test_aux_formulas_fml_ids = sdk2.parameters.List(
                'Geo suggest aux formulas fml-ids list (experimental)',
                default=None,
                required=False,
                value_type=sdk2.parameters.String)
            test_aux_experiments = sdk2.parameters.List(
                'Aux experiments list (experimental)',
                default=None,
                required=False,
                value_type=sdk2.parameters.String)
            test_custom_flags = sdk2.parameters.Dict(
                'Custom flags list (experimental)',
                default=None,
                required=False,
                value_type=sdk2.parameters.String)
            test_custom_cmdl = sdk2.parameters.String(
                'Custom command line',
                default=None,
                required=False)

        with sdk2.parameters.Group('Dolbilka parameters') as dolbilka_params:
            dolbilo_plan = sdk2.parameters.Resource(
                'Plan',
                resource_type=resources.GEO_SUGGEST_WEBDAEMON_PLAN,
                required=True)
            params = dolbilka.DolbilkaExecutor2.Parameters

        lunapark_param = tank_executor.LunaparkPlugin.Parameters
        offline_param = tank_executor.OfflinePlugin.Parameters

    class Context(sdk2.Task.Context):
        runs = {'reference': [], 'test': []}
        stats = {}

    def on_execute(self):
        self._init_virtualenv()

        reference_geosuggest = component.GeoSuggestDaemonWrapper(
            daemon_path=sdk2.ResourceData(self.Parameters.reference_daemon).path,
            config_path=get_resource_path_if_exists(self.Parameters.reference_config),
            data_dir=sdk2.ResourceData(self.Parameters.reference_data).path,
            logs_dir=self.log_path('geosuggestd_reference'),
            log_level=self.Parameters.reference_log_level,
            aux_formula_path=get_resource_path_if_exists(self.Parameters.reference_aux_formula),
            aux_formulas_fml_ids=self.Parameters.reference_aux_formulas_fml_ids,
            aux_experiments=self.Parameters.reference_aux_experiments,
            custom_flags=self.Parameters.reference_custom_flags,
            isolated_mode=self.Parameters.isolated_mode,
            cmd_line=self.Parameters.reference_custom_cmdl,
        )
        test_geosuggest = component.GeoSuggestDaemonWrapper(
            daemon_path=sdk2.ResourceData(self.Parameters.test_daemon).path,
            config_path=get_resource_path_if_exists(self.Parameters.test_config),
            data_dir=sdk2.ResourceData(self.Parameters.test_data).path,
            logs_dir=self.log_path('geosuggestd_test'),
            log_level=self.Parameters.test_log_level,
            aux_formula_path=get_resource_path_if_exists(self.Parameters.test_aux_formula),
            aux_formulas_fml_ids=self.Parameters.test_aux_formulas_fml_ids,
            aux_experiments=self.Parameters.test_aux_experiments,
            custom_flags=self.Parameters.test_custom_flags,
            isolated_mode=self.Parameters.isolated_mode,
            cmd_line=self.Parameters.test_custom_cmdl,
        )

        self.Context.runs['reference'] = []
        self.Context.runs['test'] = []

        with started(reference_geosuggest):
            with started(test_geosuggest):
                reference_geosuggest.wait()
                test_geosuggest.wait()

                for session in xrange(self.Parameters.dolbilka_executor_sessions):
                    session_no = session + 1  # 1-based
                    self._shoot_and_save_result(reference_geosuggest, 'reference', session_no)
                    self._shoot_and_save_result(test_geosuggest, 'test', session_no)

        self.Context.stats = self._aggregate_stats(self.Context.runs)

    def _get_jinja2_env(self):
        template_path = sdk2.path.Path(__file__).resolve().parent
        return jinja2.Environment(loader=jinja2.FileSystemLoader(str(template_path)))

    @sdk2.header()
    def header(self):
        if self.Context.stats:
            return self._get_jinja2_env().get_template('header.html').render(stats=self.Context.stats)
        return None

    @sdk2.footer()
    def footer(self):
        return self._get_jinja2_env().get_template('footer.html').render(runs=self.Context.runs)

    def _init_virtualenv(self):
        tank_executor.TankExecutor2.init_virtualenv(self)

    def _shoot_and_save_result(self, target, name, session_no):
        result = self._dolbilo_shoot(target, self.Parameters.dolbilo_plan, name, session_no)
        self.Context.runs[name].append(result)
        # to show current results in footer
        self.Context.save()
        if not target.running:
            raise common.errors.TaskError('Geosuggest {} daemon is dead'.format(name))

    def _dolbilo_shoot(self, target, plan_resource, name, session_no):
        executor = tank_executor.TankExecutor2(
            tank_executor.DolbiloPlugin(self, plan_resource, 'localhost', target.port),
            *self._default_plugins()
        )
        stats_resource, artifacts_dir = self._shoot(executor, name, session_no)

        # new stats
        new_stats_results = tank_executor.DolbiloPlugin.get_stats(artifacts_dir)
        report_path = sdk2.path.Path(tank_executor.TankExecutor2.get_report_path(artifacts_dir)).relative_to(stats_resource.path.resolve())
        new_stats_results.update(self._memory_usage(target))

        lunapark_url = tank_executor.LunaparkPlugin.get_job_url(self)
        if lunapark_url:
            new_stats_results['report_url'] = lunapark_url
        else:
            new_stats_results['report_url'] = '{}/{}'.format(stats_resource.http_proxy, report_path)

        return new_stats_results

    def _default_plugins(self):
        return [
            tank_executor.JsonReportPlugin(self),
            tank_executor.LunaparkPlugin(self),
            tank_executor.TelegrafPlugin(self),
            tank_executor.OfflinePlugin(self)
        ]

    def _shoot(self, executor, instance_name, session_no):
        description = '{}, session #{}'.format(instance_name, session_no)
        stats_dir = 'fire.{}.{}'.format(instance_name, session_no)
        stats_resource = tank_resources.YANDEX_TANK_LOGS(self, description, stats_dir)
        stats_resource_data = sdk2.ResourceData(stats_resource)
        stats_resource_data.path.mkdir(0o755, parents=True, exist_ok=True)
        work_dir = str(stats_resource_data.path)

        # Note: cgroup is too restrictive to obtain maximum possible rps
        artifacts_dir = executor.fire(
            work_dir,
            job_name=description,
            cgroup=None,
            autostop_expected=False,
        )
        return stats_resource, artifacts_dir

    def _memory_usage(self, target):
        info = process.get_process_info(target.process.pid, ['rss', 'vsz'], ignore_errors=False)
        return {
            'rss': int(info['rss']),
            'vsz': int(info['vsz'])
        }

    def _aggregate_single_instance_stats(self, instance_runs):
        stats = {}
        # Average top half to avoid down outliers and average rps jumps over time.
        rps = [run['shooting.rps_0.5'] for run in instance_runs]
        middle = len(rps) // 2
        avr_count = len(rps) - middle
        stats['max_requests_per_sec'] = round(sum(sorted(rps)[middle:]) / avr_count, 2) if avr_count else 0.
        stats['min_latency_50'] = min([run['shooting.latency_0.5'] for run in instance_runs])
        stats['min_latency_95'] = min([run['shooting.latency_0.95'] for run in instance_runs])
        stats['min_latency_99'] = min([run['shooting.latency_0.99'] for run in instance_runs])
        stats['response_size_50'] = mean([run['shooting.response_size_0.5'] for run in instance_runs])
        stats['response_size_95'] = mean([run['shooting.response_size_0.95'] for run in instance_runs])
        stats['response_size_max'] = mean([run['shooting.response_size_1'] for run in instance_runs])
        stats['max_memory_rss'] = max([run['rss'] for run in instance_runs])
        stats['max_memory_vsz'] = max([run['vsz'] for run in instance_runs])
        stats['min_fail_rate'] = min([1. * run['shooting.errors'] / run['dumper.total_requests'] for run in instance_runs])
        return stats

    def _aggregate_stats(self, runs):
        stats = {}
        reference = self._aggregate_single_instance_stats(runs['reference'])
        test = self._aggregate_single_instance_stats(runs['test'])
        for param in set(reference) & set(test):
            stats[param] = {
                'reference': reference[param],
                'test': test[param],
                'difference': {
                    'absolute': test[param] - reference[param],
                    'percentage': delta_percent(reference[param], test[param]),
                }
            }
        return stats
