import logging
import os
import subprocess

from datetime import datetime, timedelta
from sandbox import sdk2
from sandbox.projects.geosuggest import resources
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.paths import get_logs_folder


class GeosuggestPreparePOVTables(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 8192
        environments = (environments.PipEnvironment('yandex-yt', use_wheel=True),)

    class Parameters(sdk2.task.Parameters):
        kill_timeout = 4 * 60 * 60
        geosuggest_binaries = sdk2.parameters.Resource('Geosuggest binaries for sandbox',
                                                       resource_type=resources.GEO_SUGGEST_SANDBOX_BIN,
                                                       required=True)
        geocoder_export = sdk2.parameters.String('YT path to the geocoder export',
                                                 required=True,
                                                 default_value='//home/maps/geocoder/geosrc/latest_state')
        pov = sdk2.parameters.String('POV to use',
                                     default_value='RU')
        output_dir = sdk2.parameters.String('YT path to the output dir',
                                            required=True)
        ttl_days = sdk2.parameters.Integer('Build artifacts TTL in days',
                                           default_value=3)
        yt_cluster = sdk2.parameters.String('YT cluster',
                                            default_value='hahn')
        yt_pool = sdk2.parameters.String('YT pool',
                                         default_value='geosearch_low_priority')

    def get_binary(self):
        return str(sdk2.ResourceData(self.Parameters.geosuggest_binaries).path)

    def make_dir(self, parent_dir):
        import yt.wrapper as yt

        yt.config['token'] = self.yt_token
        yt.config['proxy']['url'] = self.Parameters.yt_cluster
        dir_path = os.path.join(parent_dir, 'geosuggest_prepared_pov_{}_{}'.format(self.Parameters.pov, self.id))

        yt.mkdir(dir_path, recursive=None)
        if self.Parameters.ttl_days is not None:
            ts = datetime.now()
            ts += timedelta(days=self.Parameters.ttl_days)
            yt.set_attribute(dir_path, 'expiration_time', ts.isoformat())

        return dir_path

    def run(self):
        proc_env = os.environ.copy()
        self.yt_token = sdk2.Vault.data('GEO_SUGGEST', 'yt_token')
        proc_env['YT_TOKEN'] = self.yt_token
        proc_env['YT_POOL'] = self.Parameters.yt_pool

        binary = os.path.join(self.get_binary(), 'prepare_pov_tables')
        output = self.make_dir(self.Parameters.output_dir)
        cmd = [
            binary,
            '--geosrc', self.Parameters.geocoder_export,
            '--pov', self.Parameters.pov,
            '--proxy', self.Parameters.yt_cluster,
            '--output', output
        ]

        logging.info('Running: {}'.format(cmd))
        log_file_path = os.path.join(get_logs_folder(), 'prepare_pov_tables.log')
        with open(log_file_path, 'w') as log_file:
            try:
                subprocess.check_call(cmd,
                                      env=proc_env,
                                      stdout=log_file,
                                      stderr=subprocess.STDOUT)
            except subprocess.CalledProcessError:
                logging.exception('%s command failed' % cmd)
                raise SandboxTaskFailureError('Building toponyms index data failed')
        return output

    def on_execute(self):
        self.Context.prepared_pov_tables = self.run()
