import logging
import time

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects import resource_types
from sandbox.projects.geosuggest.resources import MAPS_GEO_SUGGEST_ORG_MERGED_BIN
from sandbox.projects.geosuggest.common.tasks import (
    IndexerHostParameter, CommonProxyExecutableParameter, CommonProxyConfigParameter,
)
from sandbox.projects.common.utils import get_or_default
from sandbox.projects.geosuggest.UpdateMapsGeoSuggestSaasOrgs import (
    MergedOrgsParameter, UpdateMapsGeoSuggestSaasOrgs,
)
from sandbox.projects.geosuggest.UpdateMapsGeoSuggestSaasToponyms import (
    ToponymsParameter, UpdateMapsGeoSuggestSaasToponyms,
)


class Shard(object):

    def __init__(self, release, indexer_host):
        self.release = release
        self.indexer_host = indexer_host


class TaskProxy(object):

    def get_description(self, shard):
        raise NotImplementedError

    @property
    def resource_type(self):
        raise NotImplementedError

    @property
    def resource_parameter_name(self):
        raise NotImplementedError

    @property
    def task_type(self):
        raise NotImplementedError

    @property
    def kill_timeout(self):
        raise NotImplementedError


class IndexOrgsTaskProxy(TaskProxy):

    def get_description(self, shard):
        return 'Updating {} organization index'.format(shard.release)

    @property
    def resource_type(self):
        return MAPS_GEO_SUGGEST_ORG_MERGED_BIN

    @property
    def resource_parameter_name(self):
        return MergedOrgsParameter.name

    @property
    def task_type(self):
        return UpdateMapsGeoSuggestSaasOrgs.type

    @property
    def kill_timeout(self):
        return UpdateMapsGeoSuggestSaasOrgs.TIMEOUT


class IndexToponymsTaskProxy(TaskProxy):

    def get_description(self, shard):
        return 'Updating {} toponyms index'.format(shard.release)

    @property
    def resource_type(self):
        return resource_types.MAPS_GEO_SUGGEST_GEO_OBJECTS_WEIGHT

    @property
    def resource_parameter_name(self):
        return ToponymsParameter.name

    @property
    def task_type(self):
        return UpdateMapsGeoSuggestSaasToponyms.type

    @property
    def kill_timeout(self):
        return UpdateMapsGeoSuggestSaasToponyms.TIMEOUT


class UpdateMapsGeoSuggestSaasIndex(SandboxTask):
    """Reindex newly released geosuggest shards."""

    type = 'UPDATE_MAPS_GEO_SUGGEST_SAAS_INDEX'

    input_parameters = [
        CommonProxyExecutableParameter,
        CommonProxyConfigParameter,
    ]

    shards = [
        # Shard(release='prestable', indexer_host='saas-indexerproxy-maps-prestable.yandex.net'),
        Shard(release='stable', indexer_host='saas-indexerproxy-maps-prestable.yandex.net'),
        Shard(release='stable', indexer_host='saas-indexerproxy-maps.yandex.net'),
    ]

    tasks = [
        IndexOrgsTaskProxy(),
        IndexToponymsTaskProxy(),
    ]

    def on_execute(self):
        common_proxy_executable_id = get_or_default(self.ctx, CommonProxyExecutableParameter)
        if common_proxy_executable_id is None:
            common_proxy_executable_id = CommonProxyExecutableParameter.default_value

        common_proxy_config_id = self.ctx[CommonProxyConfigParameter.name]

        with self.memoize_stage.create_tasks:
            for shard in self.shards:
                for task in self.tasks:
                    resources = channel.sandbox.list_resources(
                        resource_type=task.resource_type,
                        status='READY',
                        attribute_name='released',
                        attribute_value=shard.release,
                        order_by='-id',
                        limit=1,
                    )
                    if not resources:
                        continue
                    resource = resources[0]

                    ts_attr = self.get_index_ts_attr(shard)
                    indexed_ts = channel.sandbox.get_resource_attribute(resource.id, ts_attr)
                    if indexed_ts:
                        logging.info('%s has been already indexed at %s', resource.id, indexed_ts)
                        continue

                    description = task.get_description(shard)

                    input_parameters = {
                        'kill_timeout': task.kill_timeout,
                        IndexerHostParameter.name: shard.indexer_host,
                        CommonProxyExecutableParameter.name: common_proxy_executable_id,
                        CommonProxyConfigParameter.name: common_proxy_config_id,
                        task.resource_parameter_name: resource.id,
                    }
                    logging.info('Launching subtask %s (%s) with parameters %s',
                                 task.task_type, description, input_parameters)

                    subtask = self.create_subtask(
                        task_type=task.task_type,
                        description=description,
                        input_parameters=input_parameters,
                    )

                    self.ctx[self.make_task_ctx_key(shard, task)] = subtask.id
                    self.ctx[self.make_resource_ctx_key(shard, task)] = resource.id

        with self.memoize_stage.wait_tasks:
            subtask_ids = []
            for shard in self.shards:
                for task in self.tasks:
                    subtask_id = self.ctx.get(self.make_task_ctx_key(shard, task))
                    if subtask_id:
                        subtask_ids.append(subtask_id)

            self.wait_tasks(
                tasks=subtask_ids,
                statuses=tuple(self.Status.Group.FINISH) + tuple(self.Status.Group.BREAK),
                wait_all=True
            )

        with self.memoize_stage.finalize:
            bad_statuses = (self.Status.EXCEPTION, self.Status.FAILURE, self.Status.TIMEOUT)

            for shard in self.shards:
                for task in self.tasks:
                    subtask_id = self.ctx.get(self.make_task_ctx_key(shard, task))
                    if not subtask_id:
                        continue

                    subtask = channel.sandbox.get_task(subtask_id)
                    if subtask.new_status in bad_statuses:
                        raise SandboxTaskFailureError('Subtask {} failed'.format(subtask_id))

                    resource_id = self.ctx[self.make_resource_ctx_key(shard, task)]
                    ts_attr = self.get_index_ts_attr(shard)
                    ts_val = str(time.time())
                    ok = channel.sandbox.set_resource_attribute(resource_id, ts_attr, ts_val)
                    assert ok

    def get_index_ts_attr(self, shard):
        return '{}_{}_indexed_timestamp'.format(shard.release, shard.indexer_host)

    def make_resource_ctx_key(self, shard, task_proxy):
        return '{}_resource'.format(self._make_common_ctx_key_prefix(shard, task_proxy))

    def make_task_ctx_key(self, shard, task_proxy):
        return '{}_task'.format(self._make_common_ctx_key_prefix(shard, task_proxy))

    def _make_common_ctx_key_prefix(self, shard, task_proxy):
        return '{}_{}'.format(shard.release, task_proxy.task_type)


__Task__ = UpdateMapsGeoSuggestSaasIndex
