#!/usr/bin/env python
# coding: utf-8

import logging
import datetime
import json
import copy

from sandbox.projects.common.nanny import const
from sandbox.projects.common.nanny.client import NannyClient
from sandbox.sandboxsdk.channel import channel


def get_nanny_client(nanny_token):
    """
    Returns NannyClient for production Nanny service.
    """

    return NannyClient(api_url=const.NANNY_API_URL, oauth_token=nanny_token)


def get_task_release_info(task_id):
    """
    Returns nanny_release_info content from task context.
    In case of success, returns parsed json content.
    Otherwise, returns None.
    """

    try:
        logging.info("Acquiring nanny release info of task #%s...", task_id)
        task = channel.sandbox.get_task(task_id)
        if task:
            nanny_release_info_str = task.ctx.get("nanny_release_info")
            if nanny_release_info_str:
                nanny_release_info = json.loads(nanny_release_info_str)
                logging.info("Nanny release info of task #%s: %s", task_id, json.dumps(nanny_release_info, indent=4))
                return nanny_release_info
            else:
                logging.info("Nanny release info not found in task #%s context", task_id)
        else:
            logging.info("Task #%s is not found", task_id)

    except Exception:
        logging.exception("Failed get release info of task %s", task_id)

    return None


def create_aggregated_nanny_release_info(nanny_release_infos, release_title=None, is_stable_release=None):
    """
    Combine several nanny release info structures, acquired from ReleaseToNannyTask.get_nanny_release_info() calls into one aggregated nanny release info structure.
    Params release_title and is_stable_release are used to update aggregated release info, if they are not None.
    """

    if not nanny_release_infos or not all(nanny_release_infos):
        raise ValueError("release_infos is empty or contains empty values")

    aggregated_info = copy.deepcopy(nanny_release_infos[0])
    spec = aggregated_info["spec"]
    sandbox_release = aggregated_info["spec"]["sandboxRelease"]

    for _, ri in enumerate(nanny_release_infos[1:], 1):
        item_sandbox_release = ri["spec"]["sandboxRelease"]
        for field_name in ["resourceTypes", "resources"]:
            sandbox_release[field_name].extend(item_sandbox_release[field_name])
        # join titles by default
        sandbox_release["title"] += " | " + item_sandbox_release["title"]
        spec["title"] += " | " + ri["spec"]["title"]
        # combine release stability by default
        if item_sandbox_release["releaseType"] != "stable":
            sandbox_release["releaseType"] = "prestable"

    # Copied from ReleaseToNannyTask.get_nanny_release_info() implementation:
    sandbox_release["creationTime"] = datetime.datetime.utcnow().isoformat() + "Z"

    if release_title is not None:
        spec["title"] = release_title
        sandbox_release["title"] = release_title

    if is_stable_release is not None:
        sandbox_release["releaseType"] = "stable" if is_stable_release else "prestable"

    return aggregated_info


def send_release_to_nanny(nanny_client, nanny_release_info):
    """
    Send given release to Nanny using provided client.
    In case of success, returns release id and release link.
    """

    logging.info("Sending release to Nanny, release payload: %s", json.dumps(nanny_release_info, indent=4))

    result = nanny_client.create_release2(nanny_release_info)
    logging.info("Release has been sent to Nanny")
    release_id = result["value"]["id"]

    release_link = const.RELEASE_REQUEST_TEMPLATE.format(
        nanny_api_url=const.NANNY_API_URL,
        release_request_id=release_id,
    )
    logging.info("Nanny release_id: %s, release_link: %s", release_id, release_link)
    return release_id, release_link


def send_tasks_release_to_nanny(nanny_client, task_ids_to_release, release_title, is_stable_release, releaser_task_id, releaser_task_type):
    """
    Send produced resources of given tasks to Nanny release using provided client.
    In case of success, returns release id and release link.
    """

    try:
        logging.info("Acquiring nanny release info of tasks: %s...", task_ids_to_release)
        nanny_release_infos = [get_task_release_info(task_id) for task_id in task_ids_to_release]
        if all(nanny_release_infos):
            logging.info("Aggregating releases info into one...")
            aggregated_nanny_release_info = create_aggregated_nanny_release_info(nanny_release_infos, release_title, is_stable_release)
            if releaser_task_id:
                aggregated_nanny_release_info["spec"]["sandboxRelease"]["taskId"] = str(releaser_task_id)
            if releaser_task_type:
                aggregated_nanny_release_info["spec"]["sandboxRelease"]["taskType"] = str(releaser_task_type)
            return send_release_to_nanny(nanny_client, aggregated_nanny_release_info)

        logging.info("Nanny release info not found for one or more tasks, release skipped")

    except Exception:
        logging.exception("Failed send release of tasks %s to Nanny", task_ids_to_release)

    return None, None


def get_active_snapshot_resource_ids(nanny_client, service_id):
    """
    Returns active snapshot resource ids of Nanny service using provided client.
    In case of success, returns dictionary "resource_type" -> "resource_id".
    In case of any error, returns None.
    """

    try:
        logging.info("Requesting current state of service '%s'...", service_id)
        current_state = nanny_client.get_service_current_state(service_id)

        snapshots = current_state["content"]["active_snapshots"]
        for snapshot in snapshots:
            logging.info("Found snapshot %s in state %s", snapshot["snapshot_id"], snapshot["state"])
            if snapshot["state"] in ["ACTIVE", "ACTIVATING"]:
                snapshot_id = snapshot["snapshot_id"]
                logging.info("Active snapshot id: %s, requesting snapshot attributes...", snapshot_id)
                runtime_attrs = nanny_client.get_history_runtime_attrs(snapshot_id)

                sandbox_files = runtime_attrs["content"]["resources"]["sandbox_files"]
                logging.info("Sandbox files of snapshot %s: %s", snapshot_id, json.dumps(sandbox_files, indent=4))
                resources = {item["resource_type"]: item["resource_id"] for item in sandbox_files}
                logging.info("Active resource ids for service '%s': %s", service_id, resources)
                return resources

        logging.info("No active snapshot found for service '%s'", service_id)

    except Exception:
        logging.exception("Failed to get active snapshot resource ids for service '%s'", service_id)

    return None


def update_service_resources(nanny_client, service_id, resources_to_update, new_snapshot_title=None):
    logging.info("Update resources of Nanny service '%s', title '%s', resources to update: \n%s", service_id, new_snapshot_title, "\n".join([str(r) for r in resources_to_update]))
    for rtu in resources_to_update:
        if not all([rtu.get(name) for name in ["task_type", "task_id", "resource_type", "resource_id"]]):
            raise ValueError("One of required fields in resource to update '{}' is absent".format(rtu))

    resources_data = nanny_client.get_service_resources(service_id)
    logging.info("Active resources of service: %s", json.dumps(resources_data, indent=4))

    has_changes = False
    for sf in resources_data["content"]["sandbox_files"]:
        for rtu in resources_to_update:
            if sf["resource_type"] == rtu["resource_type"]:
                if sf["resource_id"] != str(rtu["resource_id"]):
                    logging.info("Changing %s: task_id %s -> %s, resource_id %s -> %s", sf["resource_type"], sf["task_id"], rtu["task_id"], sf["resource_id"], rtu["resource_id"])
                    sf["resource_id"] = str(rtu["resource_id"])
                    sf["task_id"] = str(rtu["task_id"])
                    has_changes = True
                break

    if has_changes:
        old_snapshot_id = resources_data["snapshot_id"]
        update_data = {
            "content": resources_data["content"],
            "comment": new_snapshot_title if new_snapshot_title else " + ".join([r["resource_type"] + " " + str(r["resource_id"]) for r in resources_to_update]),
            "snapshot_id": old_snapshot_id,
        }
        logging.info("Updating service resources, update data payload: %s", json.dumps(update_data, indent=4))
        update_result = nanny_client.update_service_resources(service_id, update_data)
        logging.info("Service update result: %s", json.dumps(update_result, indent=4))

        new_snapshot_id = update_result['runtime_attrs']['_id']
        logging.info("Old snapshot id: %s", old_snapshot_id)
        logging.info("New snapshot id: %s", new_snapshot_id)
        return new_snapshot_id

    logging.info("No changes were found, service resources update is skipped")
    return None


def set_snapshot_to_prepared_state(nanny_client, service_id, snapshot_id):
    """
    Set given snapshot's state to "PREPARED", if it is not active or prepared.
    """

    try:
        logging.info("Setting snapshot '%s' of service '%s' to prepared state...", snapshot_id, service_id)
        current_state = nanny_client.get_service_current_state(service_id)

        snapshots = current_state["content"]["active_snapshots"]
        for snapshot in snapshots:
            logging.info("Found snapshot %s in state %s", snapshot["snapshot_id"], snapshot["state"])
            if snapshot["snapshot_id"] == snapshot_id:
                if snapshot["state"] in ["ACTIVE", "ACTIVATING", "PREPARED", "PREPARING"]:
                    logging.info("Snapshot '%s' is already in active or prepared state, skipping set state", snapshot_id)
                    return

        logging.info("Changing snapshot state...")
        result = nanny_client.set_snapshot_state(service_id, snapshot_id, "PREPARED", "Deploy from Sandbox")
        logging.info("Set snapshot '%s' of service '%s' state result: %s", snapshot_id, service_id, json.dumps(result, indent=4))

    except Exception:
        logging.exception("Failed to set snapshot '%s' of service '%s' to prepared state", snapshot_id, service_id)
