import os
import json

import sandbox.sandboxsdk.sandboxapi as sandboxapi
from sandbox.sandboxsdk.parameters import LastReleasedResource, SandboxStringParameter, SandboxBoolParameter

from sandbox.projects import resource_types
from sandbox.projects.common.utils import get_or_default
from sandbox.projects.geosuggest.common import pypy
from sandbox.projects.geosuggest import resources as gsr
from sandbox.projects.geobase.Geodata4BinStable import resource as gb4r
from sandbox.projects.geobase.Geodata5BinStable import resource as gb5r


class BackaExportParameter(LastReleasedResource):
    name = 'backa_export'
    description = 'Backa export'
    required = True
    resource_type = resource_types.MAPS_DATABASE_BUSINESS_SOURCE_ARCHIVE


class GeoSuggestDataBuilderParameter(LastReleasedResource):
    name = 'data_builder'
    description = 'Geo suggest data builder executable files'
    required = True
    resource_type = gsr.MAPS_GEO_SUGGEST_DATA_BUILDER


class SuggestBuilderParameter(LastReleasedResource):
    name = 'suggest_builder'
    description = 'Executable from quality/trailer/suggest/data_builder'
    resource_type = gsr.SUGGEST_DATA_BUILDER


class OrgWeightInfoParameter(LastReleasedResource):
    name = 'weight_info'
    description = 'Generated org weight-info file'
    required = True
    resource_type = gsr.MAPS_GEO_SUGGEST_ORG_WEIGHT_INFO


class GeoSuggestWeightsTableStringParameter(SandboxStringParameter):
    name = 'weight_info2'
    description = 'Path to alternative YT table with weights only'
    required = False
    default_value = '//home/qreg/geosuggest/production_data/static_weights/weights_orgs_region'


class WeightsYTTableParameter(LastReleasedResource):
    name = 'weights_yt_table'
    description = 'YT table with weights'
    required = False
    resource_type = gsr.MAPS_GEO_SUGGEST_WEIGHTS_YT_TABLE


class WeightsOsmYTTableParameter(LastReleasedResource):
    name = 'weights_osm_yt_table'
    description = 'YT table with weights of OSM objects'
    required = False
    resource_type = gsr.MAPS_GEO_SUGGEST_WEIGHTS_OSM_YT_TABLE


class GeoSuggestTrainingPoolTableParameter(LastReleasedResource):
    name = 'training_pool_table'
    description = 'YT table with training data and parameters'
    required = False
    resource_type = gsr.MAPS_GEO_SUGGEST_TRAINING_POOL_YT_TABLE


class Org2AddrNotJoined(LastReleasedResource):
    name = 'geosuggest_org_2_addr_not_joined'
    description = 'Generated companies2.not_joined'
    required = True
    resource_type = gsr.MAPS_GEO_SUGGEST_ORG_2_ADDR_NOT_JOINED


class AdditionalEnvParameter(SandboxStringParameter):
    name = 'geosuggest_additional_env_parameter'
    description = 'Additional environment variables that will be past to subprocesses (format: json dictionary with strings)'
    required = False
    default = ""

    @classmethod
    def ApplyToEnviron(cls, ctx):
        additional_env = get_or_default(ctx, AdditionalEnvParameter)
        env = {}
        if additional_env != "":
            envs = json.loads(additional_env)
            for key in envs:
                os.environ[key] = str(envs[key])
                env[key] = str(envs[key])
        return env


class SandboxVaultOwner(SandboxStringParameter):
    name = 'geosuggest_sandbox_vault_owner'
    description = 'Owner for sandbox vault'
    default_value = "GEO_SUGGEST"


class SandboxVaultYtTokenName(SandboxStringParameter):
    name = 'geosuggest_sandbox_vault_name'
    description = 'Name of vault item for YT token'
    default_value = 'yt_token'


class SandboxVaultWeatherApiTokenName(SandboxStringParameter):
    name = 'geosuggest_sandbox_vault_weather_api_token'
    description = 'Name of vault item for weather api token'
    default_value = 'weather_api_token'


class SandboxVaultNannyTokenName(SandboxStringParameter):
    name = 'geosuggest_sandbox_vault_nanny_token_name'
    description = 'Name of vault item for Nanny token'
    default_value = 'nanny_token'


class GeoSuggestPortablePypy(LastReleasedResource):
    name = 'geosuggest_portable_pypy'
    description = 'Portable pypy'
    resource_type = resource_types.PORTABLE_PYPY


class GeoSuggestPythonInterpreterParameter(SandboxStringParameter):
    name = "geosuggest_python_interpreter"
    description = "Program for interprete python scripts"
    choices = [
        ("Default", "default"),
        ("PyPy Portable", "pypy"),
    ]
    sub_fields = {
        "default": [],
        "pypy": [GeoSuggestPortablePypy.name],
    }
    default_value = "default"


class GeoSuggestPythonInterpreter(object):
    def __init__(self, task, venv):
        self.interpreter = get_or_default(task.ctx, GeoSuggestPythonInterpreterParameter)
        self.venv = venv
        if self.interpreter == 'pypy':
            self.installer = pypy.GeoSuggestPortablePyPyInstaller()
            self.installer.install(task, get_or_default(task.ctx, GeoSuggestPortablePypy))

    def pip(self, package_name):
        self.venv.pip(package_name)
        if self.interpreter == 'pypy':
            self.installer.pip(package_name)

    def get_cmd(self, args):
        if self.interpreter == 'pypy':
            return self.installer.get_cmd(args)
        else:
            return [self.venv.executable] + args


class DictResourceParameter(SandboxStringParameter):
    name = 'resource_type'
    description = 'Dict resource type'
    choices = [(rt.name, rt.name) for rt in [gsr.MAPS_GEO_SUGGEST_GEO_DICT,
                                             gsr.MAPS_GEO_SUGGEST_ORG_DICT]]
    required = True


class Geodata3BinStableParameter(LastReleasedResource):
    name = 'geodata3_bin_stable'
    description = 'Geodata 3 binary'
    resource_type = resource_types.GEODATA3BIN_STABLE


class Geodata4BinStableParameter(LastReleasedResource):
    name = 'geodata4_bin_stable'
    description = 'Geodata 4 binary'
    resource_type = gb4r.GEODATA4BIN_STABLE


class Geodata5BinStableParameter(LastReleasedResource):
    name = 'geodata5_bin_stable'
    description = 'Geodata 5 binary'
    resource_type = gb5r.GEODATA5BIN_STABLE


class GeoSuggestGeodataVersionParameter(SandboxStringParameter):
    name = "geosuggest_geodata_version"
    description = "Version of geodata binary (for selecting correct resource type)"
    choices = [
        ("geodata3 stable", "geodata3_stable"),
        ("geodata5 stable", "geodata5_stable"),
    ]
    sub_fields = {
        "geodata3_stable": [Geodata3BinStableParameter.name],
        "geodata5_stable": [Geodata5BinStableParameter.name],
    }
    default_value = "geodata5_stable"


class GeoSuggestPrepareToponymsBinParameter(LastReleasedResource):
    name = 'geosuggest_prepare_toponyms_bin_resource_id'
    description = 'Geosuggest prepare_toponyms_bin executable (see trunk/quality/trailer/suggest/services/maps/tools/prepare_toponyms_bin)'
    resource_type = gsr.MAPS_GEO_SUGGEST_PREPARE_TOPONYMS_BIN


class GeoSuggestYTPoolParameter(SandboxStringParameter):
    name = 'geosuggest_yt_pool'
    description = 'YT pool'
    default_value = None


class GeoSuggestYTProxyParameter(SandboxStringParameter):
    name = 'geosuggest_yt_proxy'
    description = 'YT proxy'
    default_value = 'hahn'


class GeoSuggestYTTablePath(SandboxStringParameter):
    name = 'geosuggest_yt_table_path'
    description = 'Table path'
    default_value = ''


class GeoSuggestYTFormat(SandboxStringParameter):
    name = 'geosuggest_yt_format'
    description = 'Format for saved data'
    default_value = '<encode_utf8=%false>json'


class GeoSuggestYTRaw(SandboxBoolParameter):
    name = 'geosuggest_yt_raw'
    description = 'Use raw format'
    default_value = False


class GeoSuggestYTTableCopyPrefix(SandboxStringParameter):
    name = 'geosuggest_yt_table_copy_prefix'
    description = 'Prefix for table copy'
    default_value = ''


class GeoSuggestBranchParameter(SandboxStringParameter):
    description = 'Geo suggest branch'
    name = 'geosuggest_branch'
    default_value = 'production'
    required = True


class ReleaseStatusSelector(SandboxStringParameter):
    choices = [
        ("None", ""),
        ("STABLE", sandboxapi.RELEASE_STABLE),
        ("PRESTABLE", sandboxapi.RELEASE_PRESTABLE),
        ("TESTING", sandboxapi.RELEASE_TESTING),
        ("UNSTABLE", sandboxapi.RELEASE_UNSTABLE),
        ("CANCELLED", sandboxapi.RELEASE_CANCELLED),
    ]
    sub_fields = {}
    default_value = ""


class GeoSuggestReleaseOnAutodeployParameter(ReleaseStatusSelector):
    name = 'geosuggest_release_on_autodeploy'
    description = 'Release data on tests successed'


class GeoSuggestExternalDaemonUrl(SandboxStringParameter):
    name = 'geosuggest_external_daemon_url'
    description = 'Geosuggest external daemon url: if not empty, use it as a template'
    default_value = ""


class GeoSuggestAutotestArguments(SandboxStringParameter):
    name = 'geosuggest_autotest_arguments'
    description = 'Additional arguments to autotest.py'
    default_value = ""


class GeoSuggestPathToAutotest(SandboxStringParameter):
    name = 'geosuggest_path_to_autotest'
    description = 'Path to autotest.py inside data builder pack'
    default_value = 'autotest'


class GeoSuggestEmailsTo(SandboxStringParameter):
    name = 'geosuggest_emails_to'
    description = 'Send email with results to'
    default_value = ''


class GeoSuggestEmailsAnyway(SandboxBoolParameter):
    name = 'geosuggest_emails_anyway'
    description = 'Send email anyway (even if it is not run from UPDATE_MAPS_GEO_SUGGEST_DATA)'


class GeoSuggestEmailsDoNotSend(SandboxBoolParameter):
    name = 'geosuggest_emails_do_not_send'
    description = 'Do not send email with task results'
    default_value = False


class GeoSuggestObjectsWeightPatameter(LastReleasedResource):
    name = 'geosuggest_toponyms_id'
    description = 'Geosuggest toponyms xml'
    required = True
    resource_type = gsr.MAPS_GEO_SUGGEST_GEO_OBJECTS_WEIGHT


class GeoSuggestStaticDataParameter(LastReleasedResource):
    name = 'geosuggest_static_data_id'
    description = 'Geosuggest static data'
    required = True
    resource_type = gsr.GEO_SUGGEST_STATIC_DATA


class GeoSuggestPackResourceType(SandboxStringParameter):
    name = 'geosuggest_pack_resource_type'
    description = 'Resource type of generated pack'
    default_value = 'GEO_SUGGEST_DATA'


class GeoSuggestSandboxBinParameter(LastReleasedResource):
    name = 'geosuggest_sandbox_bin'
    description = 'Geo suggest builder binaries (quality/trailer/suggest/services/maps/build/sandbox-bin)'
    required = False  # in case of input parameter absense, the latest released resource will be used
    resource_type = gsr.GEO_SUGGEST_SANDBOX_BIN
