import json
import os
import traceback

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.paths import make_folder

from sandbox.projects.geosuggest.common.utils import get_yt_table_info
from sandbox.projects.common.utils import get_or_default
from sandbox.projects.geosuggest.common.qp import download_content
from sandbox.projects.geosuggest.common.parameters import (
    SandboxVaultOwner, SandboxVaultYtTokenName
)
from sandbox.projects.geosuggest.resources import MAPS_GEO_SUGGEST_DATA_PREVIEW


# See GeoSuggestMakePreview for example of how to use it


def build_preview_yt(task, table, content=None):
    yt_table_info = get_yt_table_info(task.ctx, table['parameter'])
    yt_path = yt_table_info["table_path"]
    yt_proxy = "hahn"
    url_template = "https://yt.yandex-team.ru/{proxy}/#page=navigation&path={path}"
    url = url_template.format(proxy=yt_proxy, path=yt_path)
    if content is None:
        try:
            import yt.wrapper
            vault_owner = get_or_default(task.ctx, SandboxVaultOwner)
            vault_name = get_or_default(task.ctx, SandboxVaultYtTokenName)
            yt_token = task.get_vault_data(vault_owner, vault_name)
            client = yt.wrapper.YtClient(proxy=yt_proxy, token=yt_token)
            rows = []
            for row in client.read_table(yt_path, format='<encode_utf8=%false>json', raw=False):
                rows.append(row)
                if len(rows) >= table.get('max_rows', 5):
                    break
            content = rows
        except:
            content = 'Cannot get data: ' + str(traceback.format_exc())
    return {
        'helperName': table['name'],
        'content': {'<h3>{name}: <a href="{url}">{proxy}:{path}</a></h3>'.format(name=table['name'], url=url, proxy=yt_proxy, path=yt_path): content},
    }


def create_task_preview_resource(task, description, preview_dir):
    task.create_resource(
        description,
        preview_dir,
        MAPS_GEO_SUGGEST_DATA_PREVIEW,
    )


def get_task_preview_resource(task):
    resources = channel.sandbox.list_resources(
        MAPS_GEO_SUGGEST_DATA_PREVIEW,
        task_id=task.id,
        omit_failed=True,
        limit=1,
    )
    return resources[0] if resources else None


def prepare_data_preview(task, datas):
    preview_dir = os.path.join(task.abs_path(), 'preview')
    make_folder(preview_dir)
    for data in datas:
        filename = os.path.join(preview_dir, data['name'] + '.json')
        preview = build_preview_yt(task, data)
        with open(filename, 'w') as f:
            json.dump(preview, f)
    create_task_preview_resource(task, 'Preview for YT table', preview_dir)


def build_preview_footer(task, datas):
    preview_resource = get_task_preview_resource(task)
    if not preview_resource:
        return []
    footer = []
    for data in datas:
        content = ''
        try:
            filename = data['name'] + '.json'
            preview = json.loads(download_content(preview_resource, filename))
            content = preview['content'].values()[0]
        finally:
            footer.append(build_preview_yt(task, data, content))
    return footer
