#!/usr/bin/env python
# coding: utf-8

import collections
import json
import os
import urllib2

from sandbox.projects import resource_types
from sandbox.common.errors import WaitTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.sandboxapi import RELEASE_STABLE, RELEASE_PRESTABLE, RELEASE_TESTING, RELEASE_CANCELLED, RELEASE_STATUSES
from sandbox.sandboxsdk.task import SandboxTask


def download_content(resource, filename=None):
    url = resource.proxy_url
    if filename:
        url = os.path.join(url, filename)
    try:
        response = urllib2.urlopen(url)
        return response.read()
    except urllib2.HTTPError, e:
        if e.code == 404:
            return '<a href="{0}">{1}</a> is empty or does not exist'.format(url, filename)


class Env:
    def __init__(self, task, ctx):
        self.task = task
        self.ctx = ctx
        self.results = []
        self.current_resource = None
        self.previous_resource = None

    def passed(self):
        return all([result.get('ok', False) for result in self.results])

    def save_progress_to_context(self):
        count = int(self.task.ctx.get('geosuggest_qp_runs_count', 0))
        self.task.ctx['geosuggest_qp_{0}'.format(count)] = json.dumps(self.results)
        return count

    def add_result(self, name, type, previous, current, delta, check, ok, details=''):
        result = {
            'name': name,           # symbolic name for a test
            'type': type,           # 'q' for quality or 'p' for performance
            'previous': previous,   # previous value
            'current': current,     # generated value
            'delta': delta,         # delta or ''
            'check': check,         # what's checking
            'ok': ok,               # boolean for actual test result
            'details': details,     # any details about test or result
        }
        self.results.append(result)
        self.save_progress_to_context()


def get_file_size(path):
    return os.path.getsize(path)


def get_dir_size(path):
    return sum(os.path.getsize(f) for f in os.listdir(path) if os.path.isfile(f))


def get_file_lines_count(path):
    return 1000


def get_file_xml_tags_count(path, tag, depth=None):
    return 1000


def check_file_size_limit(env, prev_fname, curr_fname, limit, type='p'):
    previous = get_file_size(prev_fname)
    current = get_file_size(curr_fname)
    delta = ''
    env.add_result(
        'size({0})'.format(os.path.basename(curr_fname)),
        type,
        previous,
        current,
        delta,
        'size < {limit}'.format(limit=limit),
        current < limit,
    )


def check_file_size_delta(env, prev_fname, curr_fname, limit, type='q'):
    previous = get_file_size(prev_fname)
    current = get_file_size(curr_fname)
    delta = current - previous
    env.add_result(
        'Δsize({0})'.format(os.path.basename(curr_fname)),
        type,
        previous,
        current,
        delta,
        'Δsize < {limit}'.format(limit=limit),
        delta < limit,
    )


def check_file_size_percent(env, prev_fname, curr_fname, percent, type='q'):
    previous = get_file_size(prev_fname)
    current = get_file_size(curr_fname)
    delta = 100 * (current - previous) / max(float(previous), 1.0)
    env.add_result(
        'Delta size of {0}'.format(os.path.basename(curr_fname)),
        type,
        previous,
        current,
        '%.3g%%' % delta,
        '< {0}%'.format(percent),
        abs(delta) < percent,
    )


def check_dir_size_percent(env, prev_dir, curr_dir, percent, type='q'):
    previous = get_dir_size(prev_dir)
    current = get_dir_size(curr_dir)
    delta = 100 * (current - previous) / max(float(previous), 1.0)
    env.add_result(
        'Delta size of {0}'.format(os.path.basename(os.path.normpath(curr_dir)) + "/"),
        type,
        previous,
        current,
        '%.3g%%' % delta,
        '< {0}%'.format(percent),
        abs(delta) < percent,
    )


def check_file_lines(env, filename, limit, type='p'):
    previous = get_file_lines_count(env.get_previous_path(filename))
    current = get_file_lines_count(env.get_current_path(filename))
    delta = ''
    env.add_result(
        'lines({filename})'.format(filename=filename),
        type,
        previous,
        current,
        delta,
        'lines < {limit}'.format(limit=limit),
        current < limit,
    )


def check_file_lines_delta(env, filename, limit, type='p'):
    previous = get_file_lines_count(env.get_previous_path(filename))
    current = get_file_lines_count(env.get_current_path(filename))
    delta = current - previous
    env.add_result(
        'Δlines({filename})'.format(filename=filename),
        type,
        previous,
        current,
        delta,
        'Δlines < {limit}'.format(limit=limit),
        delta < limit,
    )


def check_file_xml_tags_count(env, filename, limit, type='p'):
    previous = get_file_xml_tags_count(env.get_previous_path(filename))
    current = get_file_xml_tags_count(env.get_current_path(filename))
    delta = ''
    env.add_result(
        'tags({filename})'.format(filename=filename),
        type,
        previous,
        current,
        delta,
        'tags < {limit}'.format(limit=limit),
        current < limit,
    )


def check_file_xml_tags_count_delta(env, filename, limit, type='p'):
    previous = get_file_xml_tags_count(env.get_previous_path(filename))
    current = get_file_xml_tags_count(env.get_current_path(filename))
    delta = ''
    env.add_result(
        'tags({filename})'.format(filename=filename),
        type,
        previous,
        current,
        delta,
        'tags < {limit}'.format(limit=limit),
        delta < limit,
    )


class GeoSuggestQPTester():
    def __init__(self, task, ctx):
        self.qp_passed = True
        self.task = task
        self.env = Env(task, ctx)

    @classmethod
    def is_qp_task(cls, task_class):
        """Checks that task_class has own QP tests and so doesn't need
           to be released from external code"""
        return issubclass(task_class, GeoSuggestQPTask)

    def create_branched_release(self, task_id, branch, status, wait=False):
        rs = None
        if branch == '' or branch == 'production':
            if status in RELEASE_STATUSES:
                rs = status
        if rs is not None:
            if self.task.id == task_id:
                self.task.mark_released_resources(rs, ttl="inf")
            else:
                try:
                    self.task.create_release(task_id, status=rs)
                    if wait:
                        releases = channel.sandbox.list_releases(task_id=task_id, release_status=rs)
                        if not releases:
                            self.task.wait_tasks(tasks=[task_id], statuses=(self.task.Status.RELEASED,) + tuple(self.task.Status.Group.BREAK), wait_all=True)
                except WaitTask:
                    raise
                except:
                    pass
        else:
            resources = channel.sandbox.list_resources(task_id=task_id, limit=None)
            if resources:
                for resource in resources:
                    channel.sandbox.set_resource_attribute(resource, 'geosuggest_branch', branch)
                    channel.sandbox.set_resource_attribute(resource, 'geosuggest_release', status)

    def run_qp_tests(self, cls, resource, resource_type, env):
        curr = self.task.sync_resource(resource.id)
        prev_resource = None
        stables = channel.sandbox.list_resources(resource_type, attribute_name='released', attribute_value=RELEASE_STABLE, limit=2)
        if stables:
            prev_resource = stables[0]
            # if current resource is last and already has status STABLE,
            # we still can run qp tests by picking next STABLE resource
            if prev_resource.id == resource.id:
                prev_resource = stables[1]
        # TODO merge stables and prestables and sort it by timestamp
        prestables = channel.sandbox.list_resources(resource_type, attribute_name='released', attribute_value=RELEASE_PRESTABLE, limit=2)
        if prestables:
            if prev_resource is None:
                prev_resource = prestables[0]
            elif prev_resource.timestamp < prestables[0].timestamp:
                prev_resource = prestables[0]
        if prev_resource is not None:
            prev = self.task.sync_resource(prev_resource.id)
            env.current_resource = resource
            env.previous_resource = prev_resource
            cls.on_qp_tests(self.task, resource_type, prev, curr, env)

    def test(self, cls, prev, curr):
        """
            Run tests on generated resources

            Parameters:
                cls   -- class of testing task (cls.on_qp_tests will be called)
                task  -- actual task inside we test (may not be instance of cls)
            Return values:
                True  -- tests passed
                False -- tests failed
        """
        if self.is_qp_task(cls):
            # TODO some logic here
            self.qp_passed = True
            resources = channel.sandbox.list_resources(task_id=curr, limit=None)
            if resources:
                for resource in resources:
                    resource_type = None
                    try:
                        resource_type = getattr(resource_types, resource.type)
                    except:
                        pass
                    if resource_type is not None and resource_type.releasable:
                        self.run_qp_tests(cls, resource, resource_type, self.env)
            self.qp_passed = self.env.passed()
            self.task.ctx['geosuggest_qp_runs_count'] = self.env.save_progress_to_context() + 1
        return self.qp_passed


class GeoSuggestQPTask(SandboxTask):
    qp_passed = True

    @classmethod
    def on_qp_tests(cls, task, rt, prev, curr, env):
        """
            Run tests on generated resources

            THIS METHOD SHOULD BE OVERRIDED IN SUBCLASSES

            Parameters:
                cls   -- self class
                task  -- task for routins (create_release for example)
                rt    -- resource type (always check this value)
                prev  -- directory to previous resource
                curr  -- directory to new generated resource
                env   -- instance of Env for storing tests results

            To fail QP tests this task should add failed result to env
        """
        pass

    def on_release(self, additional_parameters):
        subject = additional_parameters.get('release_subject', '')
        if 'qp' not in subject:
            # regular releasing without qp tests
            SandboxTask.on_release(self, additional_parameters)
            return
        branch = ''
        if 'branch=' in subject:
            branch = subject.split("branch=")[1].split(' ')[0]
        qp_tester = GeoSuggestQPTester(self, self.ctx)
        qp_tester.create_branched_release(self.id, branch, RELEASE_TESTING)
        qp_tester.test(self.__class__, self.id, None)
        if self.qp_passed:
            # TODO use PRESTABLE here when shardgraph will be able to pick it
            additional_parameters["release_status"] = RELEASE_STABLE
            qp_tester.create_branched_release(self.id, branch, RELEASE_STABLE)
        else:
            additional_parameters["release_status"] = RELEASE_CANCELLED
            qp_tester.create_branched_release(self.id, branch, RELEASE_CANCELLED)
        self.qp_passed = qp_tester.qp_passed

    @property
    def footer(self):
        count = int(self.ctx.get('geosuggest_qp_runs_count', 0))
        if count <= 0:
            return "<p>No qp test results found</p>"
        results = json.loads(self.ctx.get('geosuggest_qp_{0}'.format(count - 1), '[]'))
        names = collections.OrderedDict([
            ("name", "Name"),
            ("type", "Type"),
            ("previous", "Prev"),
            ("current", "Curr"),
            ("delta", "Δ"),
            ("check", "Check"),
            ("ok", "Ok?"),
            ("details", "Details")
        ])
        table = []
        for result in results:
            row = collections.OrderedDict()
            for field in names:
                value = result.get(field, '')
                if isinstance(value, str) or isinstance(value, unicode):
                    value = value.encode('utf-8')
                if field == "ok":
                    if value:
                        value = '<span style="color:#09ff09;">ok</span>'
                    else:
                        value = '<span style="color:#ff0909;">failed</span>'
                row[names[field]] = value
            table.append(row)
        return {"<h3>QP tests result:</h3>": table}
