import os
import sys
import time

from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import ResourceSelector, SandboxBoolParameter, SandboxStringParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.task import SandboxTask

from sandbox import common
from sandbox.projects import resource_types
from sandbox.projects.geosuggest.common.parameters import (
    AdditionalEnvParameter,
    GeoSuggestDataBuilderParameter,
    GeoSuggestSandboxBinParameter,
    GeoSuggestWeightsTableStringParameter,
    GeoSuggestYTPoolParameter,
    GeoSuggestYTProxyParameter,
    SandboxVaultOwner,
    SandboxVaultYtTokenName
)
from sandbox.projects.common.utils import get_or_default
from sandbox.projects.geosuggest.common import qp
from sandbox.projects.geosuggest.common.preview import prepare_data_preview, build_preview_footer
from sandbox.projects.geosuggest import resources


class BuildMapsGeoSuggestWeightInfoBase(qp.GeoSuggestQPTask):

    def get_dict_resource_id(self):
        raise NotImplementedError

    def get_script_path(self):
        raise NotImplementedError

    def get_output_resource_type(self):
        raise NotImplementedError

    def on_execute(self):
        AdditionalEnvParameter.ApplyToEnviron(self.ctx)

        weight_info_builder_path = self.get_script_path()

        dict_id = self.get_dict_resource_id()
        dict_path = self.sync_resource(dict_id)

        weight_info_gen_path = os.path.join(self.abs_path(), 'weight_info.gen.txt')

        run_process([sys.executable, weight_info_builder_path, dict_path, weight_info_gen_path],
                    outs_to_pipe=True)

        # Sync source_url with the upstream.
        source_url = channel.sandbox.get_resource_attribute(dict_id, 'source_url')

        self.create_resource(
            'Processed geo suggest dict',
            weight_info_gen_path,
            self.get_output_resource_type(),
            attributes={
                'dict_id': dict_id,
                'source_url': source_url,
            },
        )


class YtExportedOrgsRoot(SandboxStringParameter):
    name = "yt_exported_orgs_root"
    description = "Overriden exported organizations YT tables root"
    default_value = ""


class BuildMapsGeoSuggestOrgMergedBase(qp.GeoSuggestQPTask):
    type = None

    cores = 1
    required_ram = 8 * 1024  # 8 Gb
    execution_space = 128 * 1024
    disk_space = 128 * 1024

    TIMEOUT = 2 * 24 * 3600

    environment = (
        environments.PipEnvironment('yandex-yt', "0.7.34-0"),
        environments.PipEnvironment("yandex-yt-yson-bindings-skynet"),
    )

    base_input_parameters = [
        GeoSuggestDataBuilderParameter,
        GeoSuggestSandboxBinParameter,
        GeoSuggestWeightsTableStringParameter,
        GeoSuggestYTPoolParameter,
        GeoSuggestYTProxyParameter,
        SandboxVaultOwner,
        SandboxVaultYtTokenName,
        YtExportedOrgsRoot
    ]

    task_specific_properties = None  # Overriden in inherited task class

    def get_weights_yt_table_info(self):
        raise NotImplementedError

    def get_acceptable_country_codes_list(self):
        raise NotImplementedError

    def get_yt_token(self):
        vault_owner = get_or_default(self.ctx, SandboxVaultOwner)
        vault_name = get_or_default(self.ctx, SandboxVaultYtTokenName)
        return self.get_vault_data(vault_owner, vault_name)

    def get_yt_output_names(self):
        yt_prefix = self.task_specific_properties.yt_path_prefix + str(int(time.mktime(time.localtime())))  # e.g. "//home/qreg/processing/org_1531314104"
        yt_merged_table = yt_prefix + "_merged"  # e.g. "//home/qreg/processing/org_1531314104_merged"
        return yt_prefix, yt_merged_table

    def get_filters_filepath(self):
        data_builder_path = self.sync_resource(get_or_default(self.ctx, GeoSuggestDataBuilderParameter))
        return os.path.join(data_builder_path, "org", "data", "filters.txt")

    def run_merge(self, geo_suggest_sandbox_bin_path, weights_yt_table, global_factors_table_columns, yt_merged_table, weights_yt_table2):
        args = [
            os.path.join(geo_suggest_sandbox_bin_path, "merge"),
            "--yt-proxy", get_or_default(self.ctx, GeoSuggestYTProxyParameter),
            "--source-type", self.task_specific_properties.source_type,
            "--source-weights-yt-table", weights_yt_table,
            "--global-factors-table-columns", global_factors_table_columns,
            "--output-merged-yt-table", yt_merged_table
        ]

        if weights_yt_table2:
            args.extend(["--source-weights-yt-table2"] + [weights_yt_table2])

        yt_exported_orgs_root = get_or_default(self.ctx, YtExportedOrgsRoot)
        if yt_exported_orgs_root:
            args.extend(["--yt-exported-orgs-root"] + [yt_exported_orgs_root])

        acceptable_country_codes = self.get_acceptable_country_codes_list()
        if acceptable_country_codes:
            args.extend(["--acceptable-country-codes"] + [str(x) for x in acceptable_country_codes])

        run_process(args, log_prefix="1_merge", environment=self.proc_env)

    def run_make_ready(self, geo_suggest_sandbox_bin_path, out_dir, filters_file_path, yt_merged_table, yt_prefix):
        file_prefix = self.task_specific_properties.output_filename_prefix
        args = [
            os.path.join(geo_suggest_sandbox_bin_path, "make_ready"),
            "--yt-proxy", get_or_default(self.ctx, GeoSuggestYTProxyParameter),
            "--filter", filters_file_path,
            "--source-merged-yt-table", yt_merged_table,
            "--output-yt-directory", yt_prefix,
            "--output-prefix", os.path.join(out_dir, file_prefix)
        ]
        run_process(args, log_prefix="2_make_ready", environment=self.proc_env)

    def run_build_dictionary(self, suggest_prepare_data_path, bindata_branch_output_path, yt_prefix):
        out_dir = bindata_branch_output_path

        args = [
            suggest_prepare_data_path,
            "org_index",
            "-s", get_or_default(self.ctx, GeoSuggestYTProxyParameter),
            "-r", yt_prefix + "/ready",
            "-p", yt_prefix + "/pretty",
            "-g", yt_prefix + "/groups",
            "-o", out_dir,
            "--output-table", yt_prefix + "/org_bindata",
            "--file-prefix", self.task_specific_properties.output_filename_prefix
        ]
        run_process(args, log_prefix="3_suggest_prepare_data", environment=self.proc_env)

    def on_execute(self):
        self.proc_env = os.environ.copy()
        self.proc_env['YT_TOKEN'] = self.get_yt_token()
        yt_pool = get_or_default(self.ctx, GeoSuggestYTPoolParameter)
        if yt_pool:
            self.proc_env['YT_POOL'] = yt_pool

        geo_suggest_sandbox_bin_path = self.sync_resource(get_or_default(self.ctx, GeoSuggestSandboxBinParameter))
        weights_yt_table2 = self.ctx.get(GeoSuggestWeightsTableStringParameter.name)

        yt_prefix, yt_merged_table = self.get_yt_output_names()
        table_info = self.get_weights_yt_table_info()
        weights_yt_table = table_info["table_path"]
        global_factors_table_columns = table_info["global_factors_table_columns"]

        filters_file_path = self.get_filters_filepath()

        bindata_output_path = os.path.join(self.abs_path(), 'out_bin')
        os.makedirs(bindata_output_path)

        for branch in ['full']:
            bindata_branch_output_path = os.path.join(bindata_output_path, branch)
            os.mkdir(bindata_branch_output_path)

            # Step 1: merge tables of organizations, weights, chains and rubrics
            self.run_merge(geo_suggest_sandbox_bin_path, weights_yt_table, global_factors_table_columns, yt_merged_table, weights_yt_table2)

            # Step 2: prepare tsv-files from merged table
            self.run_make_ready(geo_suggest_sandbox_bin_path, bindata_branch_output_path, filters_file_path, yt_merged_table, yt_prefix)

            # Step 3: prepare binary dictionaries from tsv-files
            self.run_build_dictionary(os.path.join(geo_suggest_sandbox_bin_path, "suggest_prepare_data"), bindata_branch_output_path, yt_prefix)

        self.create_resource(
            'Geo suggest organizations merged data',
            bindata_output_path,
            self.task_specific_properties.binary_output_resource_type,
            attributes={},
        )

        prepare_data_preview(self, self.task_specific_properties.previews)

    @property
    def footer(self):
        return build_preview_footer(self, self.task_specific_properties.previews)


class IndexerHostParameter(SandboxStringParameter):
    name = 'indexer_host'
    description = 'Indexerproxy address'
    default_value = 'saas-indexerproxy-maps-prestable.yandex.net'


class CommonProxyExecutableParameter(ResourceSelector):
    name = 'common_proxy'
    description = 'Binary from search/daemons/rtyserver/services/geosuggest/cproxy'
    required = False
    resource_type = resource_types.RTYSERVER_COMMON_PROXY_GEOSUGGEST

    @common.utils.classproperty
    def default_value(cls):
        common_proxy_resources = channel.sandbox.list_resources(
            CommonProxyExecutableParameter.resource_type,
            omit_failed=True,
            limit=1,
        )
        common_proxy_resource = common_proxy_resources[0]
        return common_proxy_resource.id


class CommonProxyConfigParameter(ResourceSelector):
    name = 'common_proxy_config'
    description = 'Common proxy config template'
    requred = True
    resource_type = resources.MAPS_GEO_SUGGEST_COMMON_PROXY_CONFIG_TEMPLATE


class DebugParameter(SandboxBoolParameter):
    name = 'debug'
    description = 'Debug mode: print requests to stdout.'
    default_value = False
    required = False


class UpdateMapsGeoSuggestSaasIndex(SandboxTask):
    """
    Index a data source (organizations/toponyms) using common proxy.
    """

    type = None
    base_input_parameters = [
        IndexerHostParameter,
        CommonProxyExecutableParameter,
        CommonProxyConfigParameter,
        DebugParameter,
    ]
    required_ram = 120 * 1024  # 120 Gb

    TIMEOUT = 24 * 3600

    @property
    def common_proxy_source_type(self):
        raise NotImplementedError

    @property
    def common_proxy_processor_type(self):
        raise NotImplementedError

    def prepare_common_proxy_config_context(self, indexer_host):
        """Load all required files and return a substitution dict for CP config template."""
        is_debug = get_or_default(self.ctx, DebugParameter)
        context = {
            'indexer_host': indexer_host,
            'source_type': self.common_proxy_source_type,
            'processor_type': self.common_proxy_processor_type,
            'processor_extra': '',
            'send_type': 'json_debug' if is_debug else 'json',
        }
        return context

    def on_execute(self):
        indexer_host = get_or_default(self.ctx, IndexerHostParameter)

        common_proxy_executable_path = self.sync_resource(
            get_or_default(self.ctx, CommonProxyExecutableParameter),
        )

        common_proxy_config_template_path = self.sync_resource(
            self.ctx[CommonProxyConfigParameter.name]
        )
        with open(common_proxy_config_template_path) as f:
            data = f.read()
            try:
                common_proxy_config_template = data.decode('utf8')
            except Exception:
                raise RuntimeError(repr(data))

        config_context = self.prepare_common_proxy_config_context(indexer_host)

        common_proxy_config = common_proxy_config_template.format(**config_context)

        common_proxy_config_path = self.path('common_proxy_geo.conf')
        with open(common_proxy_config_path, 'w') as f:
            f.write(common_proxy_config.encode('utf8'))

        common_proxy_cmd = [
            common_proxy_executable_path,
            common_proxy_config_path,
        ]
        run_process(
            common_proxy_cmd,
            log_prefix='common_proxy',
        )
