import os
import logging

from sandbox import sandboxsdk
from sandbox.sandboxsdk.sandboxapi import RELEASE_STABLE, RELEASE_PRESTABLE

from sandbox.projects.common import apihelpers

from sandbox.projects import resource_types
from sandbox.projects.masstransit.resources import MASSTRANSIT_DATA_RASP
from sandbox.projects.geosuggest.resources import MAPS_GEO_SUGGEST_STABLE_RESOURCE

from sandbox.projects.geosuggest.common import parameters
from sandbox.projects.common.utils import get_or_default, sync_resource
from sandbox.sandboxsdk.channel import channel
from sandbox import common


def get_yt_client(task, proxy):
    import yt.wrapper
    client = yt.wrapper.YtClient(
        proxy=proxy,
        token=task.get_vault_data(get_or_default(task.ctx, parameters.SandboxVaultOwner),
                                  get_or_default(task.ctx, parameters.SandboxVaultYtTokenName)),
    )
    return client


def get_task_info(task_id):
    """ SDK1/SDK2 way to get task information by task id """
    api = common.rest.Client()
    task = api.task[task_id].read()
    return task


def get_resource_info(resource_id):
    """ SDK1/SDK2 way to get resource information by resource id """
    api = common.rest.Client()
    resource = api.resource[resource_id].read()
    return resource


def ensure_task_succeeded(task_id, comment=None):
    if get_task_info(task_id)["status"] != common.types.task.Status.SUCCESS:
        raise common.errors.TaskError("Task #{} failed{}".format(task_id, " (" + str(comment) + ")" if comment else ""))


def get_or_default_with_ctx_update(ctx, param_class):
    """ The same as get_or_default() + save used resource id in task contexts """
    if not ctx.get(param_class.name):
        ctx[param_class.name] = param_class.default_value
    return ctx[param_class.name]


def read_resource_file_content(resource_id, filename=None):
    path = sync_resource(resource_id)
    if filename is not None:
        path = os.path.join(path, filename)
    with open(path) as fd:
        return fd.read()


def get_yt_table_info(ctx, parameter):
    """
    Read text content of given single-file resource, parse it as "key=value" elements, and return as dictionary.
    Input file example:
        table_path=//home/qreg/temporary_table
        global_factors_table_columns=num_sessions,num_clicks
    """
    content = read_resource_file_content(get_or_default(ctx, parameter))

    info = {
        "table_path": None,
        "global_factors_table_columns": "num_sessions,num_clicks",
    }

    lines = [line.strip() for line in content.splitlines() if line.strip()]  # not empty lines only
    if not lines:
        return info

    # Legacy format
    # //home/table_path
    if lines[0].startswith("//"):
        info["table_path"] = lines[0]
        return info

    # Current format
    # table_path=//home/table_path
    for line in lines:
        kv = [part.strip() for part in line.split("=")]
        if len(kv) != 2:
            continue
        key, value = kv
        info[key] = value
    return info

# Picking last stable resource, but with some specific logic suitable
# for regular updats


def pick_latest_resource_for_geosuggest(resource_type):
    """Get the latest resource with the given release status or more stable one."""

    resource = None

    if resource_type in [
        MASSTRANSIT_DATA_RASP, resource_types.MAPS_DATABASE_GEOSRC
    ]:

        # temporary, we need to check for garden_environment attribute
        # see QREG-296 for details
        releases = channel.sandbox.list_releases(
            resource_type=resource_type,
            arch=sandboxsdk.util.system_info()['arch'],
            release_status=RELEASE_STABLE,
            limit=1000)
        garden_status = "production"
        garden_releases = [
            r
            for release in releases
            for r in release.resources
            if r.type == resource_type.name and r.attributes.get("garden_environment") == garden_status]
        resource = garden_releases[0] if garden_releases else None

    else:

        is_geosuggest_resource = resource_type.__name__.startswith("MAPS_GEO_SUGGEST_") \
            or resource_type.__name__.startswith("GEO_SUGGEST_")

        is_stable = isinstance(resource_type, MAPS_GEO_SUGGEST_STABLE_RESOURCE)

        if is_geosuggest_resource and not is_stable:
            resources = [
                apihelpers.get_last_released_resource(resource_type=resource_type, release_status=RELEASE_STABLE),
                apihelpers.get_last_released_resource(resource_type=resource_type, release_status=RELEASE_PRESTABLE),
            ]
            resources = [r for r in resources if r]
            resource = max(resources, key=lambda r: r.timestamp) if resources else None

        else:
            resources = channel.sandbox.list_resources(
                resource_type=resource_type,
                attribute_name='released',
                attribute_value=RELEASE_STABLE,
                limit=1
            )
            resource = resources[0] if resources else None

    logging.info("pick_latest_resource_for_geosuggest: resource_type: {}, acquired resource_id: {}".format(
        resource_type.__name__,
        str(resource.id) if resource else "None")
    )

    return resource


def check_if_new_resource_available(resource_type, old_resource_id):
    """Update is required if there is a resource with a new id."""

    latest_resource = pick_latest_resource_for_geosuggest(resource_type)
    if not latest_resource:
        logging.info("check_if_new_resource_available: resource_type: {}, releases not found".format(
            resource_type.__name__)
        )
        return False, None

    is_new_resource_available = str(latest_resource.id) != str(old_resource_id)
    logging.info(
        "check_if_new_resource_available: resource_type: {}, latest_resource id: {}, old_resource id: {}".format(
            resource_type.__name__, str(latest_resource.id), str(old_resource_id)
        )
    )
    return is_new_resource_available, latest_resource.id
