from __future__ import absolute_import

__author__ = 'g:geosuggest'

import errno
import logging
import os
import shlex
import time
import urllib

from . import log  # noqa
from . import request  # noqa

from sandbox import sandboxsdk
from sandbox import sdk2
from sandbox.projects.common import utils
import sandbox.common.types.client as ctc
import sandbox.common.types.task as ctt
import sandbox.projects.common.dynamic_models.download as dynamic_models

from sandbox.projects.geosuggest.component.daemonconf import (
    make_isolated_config,
    make_debug_personalization_config
)
import sandbox.projects.common.search.components.component as components_common
import sandbox.projects.geosuggest.resources as resources


_logger = logging.getLogger(__name__)


DEFAULT_GEO_SUGGEST_START_TIMEOUT = 60 * 60
"""
    Geo suggest default start timeout in seconds.
"""

DEFAULT_GEO_SUGGEST_SHUTDOWN_TIMEOUT = 2 * 60
"""
    Geo suggest default shutdown timeout in seconds.
"""

FML_URL = 'https://fml.yandex-team.ru/download/computed/formula?id={}&file=matrixnet.info'
ITS_FILE = 'its'
MODELS_DIR = 'models'


def create_formulas_dir():
    external_formulas_dir = sdk2.path.Path(
        sdk2.paths.get_unique_file_name(folder='',
                                        file_name=MODELS_DIR))
    external_formulas_dir.mkdir(exist_ok=False)
    return external_formulas_dir


class ComponentProfile(components_common.ProcessComponentMixin, components_common.Component):
    def __init__(self, args, log_prefix=None, shutdown_timeout=30, **exec_args):
        components_common.ProcessComponentMixin.__init__(self, args, log_prefix, shutdown_timeout, **exec_args)
        self.profile_process = None
        self.profile_args = None
        self._profile_log_prefix = 'profile ' + self._log_prefix

    def profile(self):
        if self.run_cmd_patcher is None:
            return
        self.profile_args = self.run_cmd_patcher([])
        if len(self.profile_args) > 0 and self.profile_args[-1] == '--':
            self.profile_args.pop()
        if not any(x.startswith('-e') or x.startswith('--event=') for x in self.profile_args):
            self.profile_args += ['-e', 'cycles:u']
        self.profile_process = sandboxsdk.process.run_process(
            self.profile_args + ['-p', str(self.process.pid)],
            outputs_to_one_file=False,
            log_prefix=self._profile_log_prefix,
            wait=False,
        )
        _logger.info("Profile process id: %s", self.profile_process.pid)

    def stop_profile(self):
        if self.profile_process is None:
            _logger.info("Profile process in None, nothing to stop")
            return

        _logger.info("Waiting for profile process %s termination in %s seconds", self._profile_log_prefix, self._shutdown_timeout)
        wait_for = time.time() + self._shutdown_timeout
        while time.time() < wait_for:
            if self.profile_process.poll() is not None:
                _logger.info("Process %s was successfully terminated", self._profile_log_prefix)
                return

            try:
                self.profile_process.terminate()
            except EnvironmentError as e:
                if e.errno != errno.ESRCH:
                    _logger.warning("Failed to terminate process %s: %s", self._profile_log_prefix, e)
                _logger.info("Process %s was successfully terminated", self._profile_log_prefix)
                return
            time.sleep(0.1)

        try:
            _logger.info("Process %s is still alive. Killing it.", self._profile_log_prefix)
            self.profile_process.kill()
        except EnvironmentError as e:
            if e.errno != errno.ESRCH:
                _logger.warning("Failed to kill process %s: %s", self._profile_log_prefix, e)

    def start(self):
        self._run_process()
        _logger.info("Process %s was successfully started", self._log_prefix)

    def __enter__(self):
        super(ComponentProfile, self).__enter__()
        self.profile()
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        self.stop_profile()
        super(ComponentProfile, self).__exit__(exc_type, exc_value, traceback)


class GeoSuggestDaemonWrapper(
    components_common.WaitUrlComponentMixin,
    ComponentProfile,
):
    LOG_LEVEL_DEBUG = 'd'
    LOG_LEVEL_ERROR = 'e'
    LOG_LEVEL_INFO = 'i'

    LOG_LEVEL_CHOICES = (('debug', LOG_LEVEL_DEBUG),
                         ('info', LOG_LEVEL_INFO),
                         ('error', LOG_LEVEL_ERROR))

    """
    Wrapper over `geo suggest daemon`_.

    .. _geo suggest daemon: https://a.yandex-team.ru/arc/trunk/arcadia/quality/trailer/suggest/services/maps/SuggestFcgiD/
    """

    def __init__(self,
                 daemon_path,
                 data_dir,
                 logs_dir,
                 config_path=None,
                 aux_formula_path=None,
                 aux_formulas_fml_ids=None,
                 aux_experiments=None,
                 custom_flags=None,
                 log_level=LOG_LEVEL_ERROR,
                 port=None,
                 start_timeout=DEFAULT_GEO_SUGGEST_START_TIMEOUT,
                 shutdown_timeout=DEFAULT_GEO_SUGGEST_SHUTDOWN_TIMEOUT,
                 isolated_mode=False,
                 isolated_enable_personalization=False,
                 cmd_line=None,
                 personal_data_mocks_dir=None):
        """
            :param daemon_path: path to `geosuggestd` executable
            :param data_dir: data directory
            :param logs_dir: daemon logs directory
            :param config_path: path to daemon config
            :param port: port for daemon to listen to
            :param start_timeout: daemon start timeout (in seconds)
            :param shutdown_timeout: daemon shutdown timeout (in seconds)
            :param personal_data_mocks_dir: directory with mocks for personal data
        """
        assert daemon_path.is_file()

        if port is None:
            port = components_common.try_get_free_port()

        # GEO_SUGGEST_DATA: pack/data/...
        assert data_dir.is_dir()
        if (data_dir / 'data').is_dir():
            data_dir = data_dir / 'data'

        self._logs_dir = logs_dir
        logs_dir.mkdir(exist_ok=True)

        use_personal_mocks = personal_data_mocks_dir and personal_data_mocks_dir.is_dir()

        if config_path is None:
            config_path = data_dir / 'daemon.conf'

            if isolated_mode:
                # apply a patch
                patched_path = logs_dir / 'daemon-{}.patched.conf'.format(port)
                make_isolated_config(config_path, patched_path, not isolated_enable_personalization)
                config_path = patched_path

            if use_personal_mocks:
                patched_path = logs_dir / 'daemon-{}.patched-personal.conf'.format(port)
                make_debug_personalization_config(config_path, patched_path)
                config_path = patched_path

        assert config_path.is_file()

        external_formulas_dir = None
        if aux_formula_path is not None:
            if not external_formulas_dir:
                external_formulas_dir = create_formulas_dir()
            sdk2.paths.copy_path(source=str(aux_formula_path.resolve()),
                                 destination=str(external_formulas_dir.resolve()))

        if aux_formulas_fml_ids is not None and len(aux_formulas_fml_ids) > 0:
            if not external_formulas_dir:
                external_formulas_dir = create_formulas_dir()
            for formula_fml_id in aux_formulas_fml_ids:
                dynamic_models.download_url(models_dir=str(external_formulas_dir.resolve()),
                                            name='{}.info'.format(formula_fml_id),
                                            url=FML_URL.format(formula_fml_id))

        its_path = None
        if custom_flags is not None and len(custom_flags) > 0:
            _logger.info('geosuggestd custom flags: {}'.format(custom_flags))

            its_data = [('replace', urllib.urlencode({k: v})) for k, v in custom_flags.items()]
            its_data = urllib.urlencode(its_data)

            _logger.info('its data: {}'.format(its_data))

            its_path = sdk2.paths.get_unique_file_name(folder='', file_name=ITS_FILE)
            with open(its_path, 'w') as f:
                f.write(its_data)
            its_path = sdk2.path.Path(its_path)

        args = [
            daemon_path,
            '-C', data_dir,
            '--port', port,
            '--level', log_level,
            '--log-prefix', logs_dir / 'log-',
            '--access-log', logs_dir / 'access-',
            '--error-log', logs_dir / 'error-'
        ]
        if external_formulas_dir:
            args.extend(['--external-formulas-dir', external_formulas_dir.resolve()])
        if use_personal_mocks:
            args.extend(['--personal-data-mocks-dir', personal_data_mocks_dir.resolve()])
        aux_experiments = list(filter(None, aux_experiments or []))
        if aux_experiments:
            args.extend(['--aux-experiments', ','.join(map(lambda e: e.strip(), aux_experiments))])
        if its_path:
            args.extend(['--its-file-path', its_path.resolve()])
        if cmd_line:
            args.extend(shlex.split(cmd_line))
        args.extend(['--', config_path.resolve()])
        args = [str(arg) for arg in args]

        _logger.info('geosuggestd args: {}'.format(args))

        self.port = port
        ComponentProfile.__init__(
            self,
            args=args,
            log_prefix='geosuggestd',
            shutdown_timeout=shutdown_timeout,
        )
        components_common.WaitUrlComponentMixin.__init__(
            self,
            url="http://localhost:{}/ping".format(self.port),
            wait_timeout=start_timeout,
        )

    @property
    def daemon_log(self):
        return self._logs_dir / 'log-daemon.log'

    @property
    def running(self):
        return (self.process is not None) and (self.process.poll() is None)


_GEO_SUGGEST_PARAMETERS_GROUP = 'Geo suggest parameters'
"""
    Group of parameters for geo suggest daemon.
"""

DEFAULT_GEO_SUGGEST_PERFORMANCE_SUBTASKS_COUNT = 5
"""
    Default number of geo suggest performance subtasks.
"""

GEO_SUGGEST_DAEMON_CTX_KEY = 'geosuggestd_resource_id'
GEO_SUGGEST_DATA_CTX_KEY = 'geosuggest_data_resource_id'
GEO_SUGGEST_CONFIG_CTX_KEY = 'geosuggest_config_resource_id'
GEO_SUGGEST_PLAN_CTX_KEY = 'geosuggest_dolbilka_plan_resource_id'
GEO_SUGGEST_START_CTX_KEY = 'geosuggest_start_timeout'
GEO_SUGGEST_SHUTDOWN_CTX_KEY = 'geosuggest_shutdown_timeout'
GEO_SUGGEST_OPTIONAL_DATA_CTX_KEY = 'geosuggest_optional_data_resource_id'
GEO_SUGGEST_OPTIONAL_PLAN_CTX_KEY = 'geosuggest_optional_dolbilka_plan_resource_id'
GEO_SUGGEST_TEST_PERFORMANCE_SUBTASKS_COUNT_CTX_KEY = 'geosuggest_test_performance_subtasks_count'


SE_TAG_I_AM_ROBOT = 'i_am_robot'
"""
    Task tag for robots, set this tag when you are going to create this task automatically. see
    `cookbook`_ for details.

    .. _cookbook: https://wiki.yandex-team.ru/sandbox/cookbook/#kakogranichitkolichestvozapushhennyxzadachopredelennogotipa
"""


class GeoSuggestDaemonParameter(sandboxsdk.parameters.ResourceSelector):
    name = GEO_SUGGEST_DAEMON_CTX_KEY
    required = True
    description = 'Geo suggest daemon, see https://nda.ya.ru/3RbPHV'
    resource_type = resources.GEO_SUGGEST_WEBDAEMON
    group = _GEO_SUGGEST_PARAMETERS_GROUP


class GeoSuggestDataParameter(sandboxsdk.parameters.ResourceSelector):
    name = GEO_SUGGEST_DATA_CTX_KEY
    required = True
    description = 'Geo suggest data, see https://nda.ya.ru/3RbPHf'
    resource_type = resources.GEO_SUGGEST_DATA
    group = _GEO_SUGGEST_PARAMETERS_GROUP


class GeoSuggestConfigParameter(sandboxsdk.parameters.ResourceSelector):
    name = GEO_SUGGEST_CONFIG_CTX_KEY
    required = False
    description = 'Geo suggest config, by default will be taken from data'
    resource_type = resources.GEO_SUGGEST_WEBDAEMON_CONFIG
    group = _GEO_SUGGEST_PARAMETERS_GROUP


class GeoSuggestDolbilkaPlanParameter(sandboxsdk.parameters.ResourceSelector):
    name = GEO_SUGGEST_PLAN_CTX_KEY
    required = True
    description = 'Geo suggest dolbilka plan, see https://nda.ya.ru/3RbRcr'
    resource_type = resources.GEO_SUGGEST_WEBDAEMON_PLAN
    group = _GEO_SUGGEST_PARAMETERS_GROUP


class GeoSuggestStartTimeoutParameter(sandboxsdk.parameters.SandboxIntegerParameter):
    name = GEO_SUGGEST_START_CTX_KEY
    description = 'Geo suggest start timeout in seconds'
    default_value = DEFAULT_GEO_SUGGEST_START_TIMEOUT
    group = _GEO_SUGGEST_PARAMETERS_GROUP


class GeoSuggestShutdownTimeoutParameter(sandboxsdk.parameters.SandboxIntegerParameter):
    name = GEO_SUGGEST_SHUTDOWN_CTX_KEY
    description = 'Geo suggest shutdown timeout in seconds'
    default_value = DEFAULT_GEO_SUGGEST_SHUTDOWN_TIMEOUT
    group = _GEO_SUGGEST_PARAMETERS_GROUP


class GeoSuggestNumberOfPerformanceSubtasks(sandboxsdk.parameters.SandboxIntegerParameter):
    name = GEO_SUGGEST_TEST_PERFORMANCE_SUBTASKS_COUNT_CTX_KEY
    description = 'Number of performance subtasks to run'
    group = _GEO_SUGGEST_PARAMETERS_GROUP
    default_value = DEFAULT_GEO_SUGGEST_PERFORMANCE_SUBTASKS_COUNT


class GeoSuggestIsolatedModeParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'isolated_mode'
    description = 'Isolated mode: do not use external services'
    group = _GEO_SUGGEST_PARAMETERS_GROUP
    default_value = False
    required = False


class GeoSuggestIsolatedEnablePersonalizationParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'isolated_enable_personalization'
    description = 'Isolated mode: ... but enable personalization'
    group = _GEO_SUGGEST_PARAMETERS_GROUP
    default_value = False
    required = False


class GeoSuggestAuxFormulaParameter(sandboxsdk.parameters.ResourceSelector):
    name = 'geosuggest_aux_formula'
    description = 'Geo suggest aux formula resource id (experimental)'
    resource_type = resources.GEO_SUGGEST_MATRIXNET_MODEL
    required = False
    default_value = None
    group = _GEO_SUGGEST_PARAMETERS_GROUP


class GeoSuggestAuxFormulasParameter(sandboxsdk.parameters.ListRepeater,
                                     sandboxsdk.parameters.SandboxStringParameter):
    name = 'geosuggest_aux_formulas'
    description = 'Geo suggest aux formulas fml-ids list (experimental)'
    required = False
    default_value = None
    group = _GEO_SUGGEST_PARAMETERS_GROUP


class GeoSuggestAuxExperimentsParameter(sandboxsdk.parameters.ListRepeater,
                                        sandboxsdk.parameters.SandboxStringParameter):
    name = 'geosuggest_aux_experiments'
    description = 'Geo suggest aux experiments list (experimental)'
    required = False
    default_value = None
    group = _GEO_SUGGEST_PARAMETERS_GROUP


class GeoSuggestCustomFlagsParameter(sandboxsdk.parameters.DictRepeater,
                                     sandboxsdk.parameters.SandboxStringParameter):
    name = 'geosuggest_custom_flags'
    description = 'Geo suggest custom flags list (experimental)'
    required = False
    default_value = None
    group = _GEO_SUGGEST_PARAMETERS_GROUP


class GeoSuggestLogLevelParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = 'geosuggest_logging_level'
    description = 'Geo suggest logging level'
    choices = GeoSuggestDaemonWrapper.LOG_LEVEL_CHOICES
    default_value = GeoSuggestDaemonWrapper.LOG_LEVEL_ERROR
    group = _GEO_SUGGEST_PARAMETERS_GROUP


class GeoSuggestUsePersonalMocksParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = 'geosuggest_use_personal_mocks'
    description = 'Launch daemon with personal mocks (experimental)'
    default_value = False


LOAD_TESTING_INPUT_PARAMETERS = (
    GeoSuggestDaemonParameter,
    GeoSuggestDataParameter,
    GeoSuggestConfigParameter,
    GeoSuggestStartTimeoutParameter,
    GeoSuggestShutdownTimeoutParameter,
    GeoSuggestIsolatedModeParameter,
    GeoSuggestIsolatedEnablePersonalizationParameter,
    GeoSuggestAuxFormulaParameter,
    GeoSuggestAuxFormulasParameter,
    GeoSuggestAuxExperimentsParameter,
    GeoSuggestCustomFlagsParameter,
    GeoSuggestLogLevelParameter
)

LOAD_TESTING_CLIENT_TAGS = (
    ctc.Tag.GENERIC
    & ctc.Tag.LINUX_XENIAL
    & ctc.Tag.INTEL_E5_2660
)
"""
For load testing we need hosts with production-like characteristics.
"""

_EXECUTION_SPACE_JITTER = 20
_REQUIRED_RAM_JITTER = 5

REQUIRED_RAM = (130 + _REQUIRED_RAM_JITTER) * 1024  # 130 Gb is the RAM limit for daemon in prod
"""
    Default geo suggest daemon RAM requirements.
"""

EXECUTION_SPACE = (92 + _EXECUTION_SPACE_JITTER) * 1024 + REQUIRED_RAM  # daemon data is about 92 Gb in size
"""
    Default geo suggest deamon disk space requirements.
"""


def copy_context_params(from_ctx, to_ctx, param_types):
    for t in param_types:
        if t.name not in from_ctx:
            continue

        assert t.name not in to_ctx
        to_ctx[t.name] = from_ctx[t.name]


def disable_notifications(ctx):
    ctx['notify_via'] = ''
    ctx['notify_if_finished'] = ''
    ctx['notify_if_failed'] = ''


def aggregate_dolbilka_ctx_fields(to, from_):
    """
        Aggregate lunapark context fields from one context to another.
    """
    to['max_rps'] = max(to.get('max_rps', 0.), from_.get('max_rps', 0.))
    to['min_fail_rate'] = min(to.get('min_fail_rate', float('Inf')), from_.get('min_fail_rate', 0.))

    for x in ('requests_per_sec', 'notfound_rates', 'memory_rss', 'memory_vsz', 'fail_rates', 'results', ):
        if x not in to:
            to[x] = []

    to['requests_per_sec'].extend(from_.get('requests_per_sec', []))
    to['notfound_rates'].extend(from_.get('notfound_rates', []))
    to['memory_rss'].extend(from_.get('memory_rss', []))
    to['memory_vsz'].extend(from_.get('memory_vsz', []))
    to['fail_rates'].extend(from_.get('fail_rates', []))
    to['results'].extend(from_.get('results', []))


class GeoSuggestDaemonTester(sandboxsdk.task.SandboxTask):

    """
    See GeoSuggestTestPerformance for detailed description.
    """

    input_parameters = LOAD_TESTING_INPUT_PARAMETERS

    client_tags = LOAD_TESTING_CLIENT_TAGS
    execution_space = EXECUTION_SPACE
    required_ram = REQUIRED_RAM
    cores = 32

    SE_TAGS = {
        SE_TAG_I_AM_ROBOT: 50,
    }

    def on_enqueue(self):
        sandboxsdk.task.SandboxTask.on_enqueue(self)
        if self.se_tag:
            sem_name = "{}/{}".format(self.type, self.se_tag)
            self.semaphores(ctt.Semaphores(
                acquires=[
                    ctt.Semaphores.Acquire(name=sem_name, capacity=self.SE_TAGS[self.se_tag])
                ]
            ))

    def start_daemon(self, custom_data_dir=None, personal_data_mocks_dir=None):
        """
        Reads parameters from context, downloads all necessary resources and
        creates a GeoSuggestDaemonWrapper instance.
        """
        daemon_path = self.sync_resource(self.ctx[GeoSuggestDaemonParameter.name])

        data_dir = custom_data_dir or self.sync_resource(self.ctx[GeoSuggestDataParameter.name])

        aux_formula_path = None
        if self.ctx.get(GeoSuggestAuxFormulaParameter.name) is not None:
            aux_formula_path = self.sync_resource(self.ctx[GeoSuggestAuxFormulaParameter.name])

        config_path = None
        if self.ctx.get(GeoSuggestConfigParameter.name, None) is not None:
            config_path = self.sync_resource(self.ctx[GeoSuggestConfigParameter.name])

        logs_dir = os.path.join(self.log_path(), 'geosuggestd')

        return GeoSuggestDaemonWrapper(
            daemon_path=sdk2.path.Path(daemon_path),
            data_dir=sdk2.path.Path(data_dir),
            logs_dir=sdk2.path.Path(logs_dir),
            config_path=sdk2.path.Path(config_path) if config_path else None,
            aux_formula_path=sdk2.path.Path(aux_formula_path) if aux_formula_path else None,
            aux_formulas_fml_ids=utils.get_or_default(self.ctx, GeoSuggestAuxFormulasParameter),
            aux_experiments=utils.get_or_default(self.ctx, GeoSuggestAuxExperimentsParameter),
            custom_flags=utils.get_or_default(self.ctx, GeoSuggestCustomFlagsParameter),
            log_level=utils.get_or_default(self.ctx, GeoSuggestLogLevelParameter),
            start_timeout=utils.get_or_default(self.ctx, GeoSuggestStartTimeoutParameter),
            shutdown_timeout=utils.get_or_default(self.ctx, GeoSuggestShutdownTimeoutParameter),
            isolated_mode=utils.get_or_default(self.ctx, GeoSuggestIsolatedModeParameter),
            isolated_enable_personalization=utils.get_or_default(self.ctx, GeoSuggestIsolatedEnablePersonalizationParameter),
            personal_data_mocks_dir=personal_data_mocks_dir
        )


class ProductionLikeRequirements(sdk2.Task.Requirements):
    # Settings for production base
    disk_space = EXECUTION_SPACE
    ram = REQUIRED_RAM

    class Caches(sdk2.Requirements.Caches):
        pass


class SingleGeosuggestParameters(sdk2.Task.Parameters):
    with sdk2.parameters.Group("Geosuggest parameters") as geosuggest_params:
        daemon = sdk2.parameters.Resource('Geosuggest daemon', resource_type=resources.GEO_SUGGEST_WEBDAEMON, required=True)
        data = sdk2.parameters.Resource('Geosuggest data', resource_type=resources.GEO_SUGGEST_DATA, required=True)
        isolated_mode = sdk2.parameters.Bool('Isolated mode', description='Do not use external services', default=False, required=False)
        with isolated_mode.value[True]:
            isolated_enable_personalization = sdk2.parameters.Bool('... enable personalization', default=False, required=False)
