from lxml import etree


class Config(object):
    def __init__(self, tree):
        self._tree = tree
        self._xpath_prefix = '/fastcgi/geosuggest'

    def remove(self, path):
        for elem in self._tree.xpath(self._xpath_prefix + path):
            elem.getparent().remove(elem)

    def override(self, path, value):
        '''
        Changes the value if the setting is present.
        Does not add a new setting if it has not been already set.
        '''
        for elem in self._tree.xpath(self._xpath_prefix + path):
            elem.text = value

    def set(self, path, value):
        '''
        Creates the setting if it is not present.
        '''
        self._get_or_create(self._xpath_prefix + path).text = str(value)

    def _get_or_create(self, path):
        cur = self._tree.getroot()
        assert cur.tag == 'fastcgi'
        for name in path.split('/')[2:]:
            nxt = cur.find(name)
            if nxt is None:
                nxt = etree.Element(name)
                cur.append(nxt)
            cur = nxt
        return cur


class MutableConfig(object):
    def __init__(self, src, dst):
        with src.open('rb') as fd:
            self.tree = etree.parse(fd)
        self.dst = dst

    def __enter__(self):
        return Config(self.tree)

    def __exit__(self, type, value, traceback):
        with self.dst.open('wb') as fd:
            self.tree.write(fd, encoding='utf-8', xml_declaration=True, pretty_print=True)


def make_debug_personalization_config(src, dst):
    with MutableConfig(src=src, dst=dst) as config:
        config.set('/debug-personal-user-id-cgi-param-enabled', '1')


def make_isolated_config(src, dst, disable_personalization=True):
    '''
    Takes a path to the geosuggest config file,
    modifies the configuration to switch off external services,
    writes the new config to a given path.
    '''
    with MutableConfig(src, dst=dst) as config:
        config.remove('/knn_hostport')
        config.remove('/geosearch-hostport')
        config.remove('/geosearch_hostports')
        config.remove('/personal-saas-kv')
        config.remove('/blackbox-address')
        config.remove('/promo-queries')

        config.override('/blackbox-check-oauth', '0')
        config.override('/saas/enable', '0')
        config.override('/request-nahodki-for-suggests', '0')
        if disable_personalization:
            config.override('/personal-yt/enable', '0')
            config.override('/personal-handler/mode', 'dummy')
            config.override('/request-rtmr-for-suggests', '0')
            config.override('/personal-api-rtmr/enable', '0')

        config.set('/erratum-address', '')
        config.set('/trusted-mode', '1')
        config.set('/debug-personal-user-id-cgi-param-enabled', '1')
