import logging
import re

HEADER = '*** {} ***'
TIME_MATCHER = re.compile(r'.*?time_guard.h:(?:\d+) (.*?): FINISH\((\d+\.\d+)s\).*')
MEMORY_MATCHER = re.compile(r'.*?memory_guard.h:(?:\d+) Finish (.*?):.*?rss_delta: (\d+\.\d+)Mb.*')

MAX_LINES = 100


def _startup_time_lines(fileobj):
    result = []

    for line in fileobj:
        m = TIME_MATCHER.match(line)
        if m:
            result.append((m.group(1), float(m.group(2))))
            if len(result) >= MAX_LINES:
                break

    result.sort(key=lambda x: -x[1])

    report = []
    for l in result:
        report.append('{:7.3f} s  {}'.format(l[1], l[0]))
    return report


def _startup_memory_lines(fileobj):
    result = []

    for line in fileobj:
        m = MEMORY_MATCHER.match(line)
        if m:
            result.append((m.group(1), float(m.group(2))))
            if len(result) >= MAX_LINES:
                break

    result.sort(key=lambda x: -x[1])

    report = []
    for line in result:
        report.append('{:10.3f} MB  {}'.format(line[1], line[0]))

    total = sum(line[1] for line in result)

    report.append('==' * 25)
    report.append('{:10.3f} MB'.format(total))
    return report


def _parse_log(title, func, path):
    report = [HEADER.format(title)]
    try:
        with path.open() as fd:
            report += func(fd)
    except Exception as e:
        logging.exception('Unable to get %s from daemon log %s', title, path)
        report += ['{}: {}'.format(type(e).__name__, e)]
    return '\n'.join(report)


def get_startup_time_stats(fileobj):
    return _parse_log('Startup Time', _startup_time_lines, fileobj)


def get_startup_memory_stats(fileobj):
    return _parse_log('Startup Memory', _startup_memory_lines, fileobj)
