from __future__ import unicode_literals

import json
import re
import six
import xmltodict

from geosuggest.proto.suggest_pb2 import Suggest
from yandex.maps.proto.suggest.suggest_pb2 import Response
from google.protobuf.json_format import MessageToDict
from google.protobuf.message import DecodeError

REQID_KEYS = ['suggest_reqid', 'server_reqid']
FAKE_REQID = 'not involved in diff'


def _set_fake_reqid(obj):
    if isinstance(obj, dict):
        for key in REQID_KEYS:
            reqid = obj.get(key)
            if isinstance(reqid, six.text_type):
                obj[key] = FAKE_REQID
        for v in obj.values():
            _set_fake_reqid(v)

    elif isinstance(obj, list):
        for v in obj:
            _set_fake_reqid(v)


def _unicodise(obj):
    if isinstance(obj, dict):
        return {_unicodise(key): _unicodise(value)
                for key, value in six.iteritems(obj)}
    elif isinstance(obj, list):
        return [_unicodise(element) for element in obj]
    elif isinstance(obj, six.binary_type):
        return obj.decode('utf-8')
    else:
        return obj


def _unpack_logids(message):
    items = message.get('item', [])
    if not isinstance(items, list):
        # single item
        items = [items]

    for item in items:
        if not isinstance(item, dict):
            continue
        log_id = item.get('log_id')
        if log_id is not None:
            try:
                log_id = json.loads(log_id)
            except json.JSONDecodeError:
                pass
            else:
                _set_fake_reqid(log_id)
                item['log_id'] = log_id


def _parse_json(blob):
    result = json.loads(blob)
    _set_fake_reqid(result)
    return result


def _parse_v0(text):
    lines = text.splitlines(False)
    if lines:
        num_tokens = lines[0].count('\t')
        if num_tokens < 6:
            return None
        for line in lines[1:]:
            if line.count('\t') != num_tokens:
                return None
    return lines


def response_as_object(blob):
    '''
    Returns geosuggest response as a parsed JSON-like Python object.
    All strings have Unicode type.
    '''
    # simple JSON
    if re.match(br'\{.*\}', blob) or re.match(br'\[.*\]', blob):
        return _parse_json(blob)

    # JSONP
    m = re.match(br'[\w\.]+\((.*)\)', blob)
    if m is not None:
        return _parse_json(m.group(1))

    # XML
    if blob.startswith(b'<?xml'):
        return xmltodict.parse(blob)

    try:
        lines = _parse_v0(blob.decode('utf-8'))
        if lines is not None:
            return lines
    except UnicodeDecodeError:
        pass

    # Protobuf v4
    try:
        message = Response.FromString(blob)
    except DecodeError:
        pass
    else:
        if message.reqid is not None:
            message.reqid = FAKE_REQID
        result = MessageToDict(message, preserving_proto_field_name=True)
        result = _unicodise(result)
        _unpack_logids(result)
        return result

    # Protobuf v3
    try:
        message = Suggest.FromString(blob)
    except DecodeError:
        pass
    else:
        result = MessageToDict(message, preserving_proto_field_name=True)
        result = _unicodise(result)
        return result

    raise ValueError('unknown format of geosuggest response')
