from sandbox import common
from sandbox import sdk2

from sandbox.common.types.misc import NotExists
import sandbox.common.types.task as ctt

from sandbox.projects import DeployNannyDashboard
from sandbox.projects.common import apihelpers
from sandbox.projects.common.nanny import nanny
from sandbox.projects.VideoSearch import VideoVideohubDumpFilesFromArcadia


class GoodsReleaseDynamicResource(nanny.ReleaseToNannyTask2, sdk2.Task):
    task = 'GOODS_RELEASE_DYNAMIC_RESOURCE'

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('FilesPaths'):
            arcadia_root_url = sdk2.parameters.String(
                'Arcadia root url',
                required=True
            )
            arcadia_paths = sdk2.parameters.List(
                'Arcadia paths',
                required=True,
            )

        with sdk2.parameters.Group('ResourceParams'):
            resource_type = sdk2.parameters.String(
                'resource_type',
                required=True
            )
            folder_name = sdk2.parameters.String(
                'folder_name',
                required=True
            )
            version_file_path = sdk2.parameters.String(
                'add version file',
                required=False
            )

        with sdk2.parameters.Group('DeployParams'):
            release_title = sdk2.parameters.String(
                'release_title',
                required=True,
            )

            nanny_dashboard_name = sdk2.parameters.String(
                'deployment_nanny_dashboard_name',
                required=True
            )
            nanny_dashboard_recipe = sdk2.parameters.String(
                'deployment_nanny_dashboard_recipe',
                required=True,
            )

            services_from_recipe = sdk2.parameters.Bool(
                'services_from_recipe',
                default_value=False,
            )
            deployment_release_status = sdk2.parameters.String(
                'deployment_release_status',
                default_value='stable',
                choices=[
                    ('stable', 'stable'),
                    ('prestable', 'prestable'),
                    ('testing', 'testing'),
                    ('unstable', 'unstable'),
                ],
            )

            vault_name = sdk2.parameters.String(
                'vault_name',
                required=True,
            )
            vault_owner = sdk2.parameters.String(
                'vault_owner',
                required=True,
            )

            semaphore_name = sdk2.parameters.String(
                'semaphore_name',
                required=True,
            )

    def on_execute(self):
        with self.memoize_stage.build_resource:
            self._build_resource()

        with self.memoize_stage.check_build_resource:
            self._check_build_resource_task()
            if not self._is_changed_resource():
                return 0

        with self.memoize_stage.deploy_resource:
            self._release_task()
            self._run_dashboard_task()

        with self.memoize_stage.check_deploy:
            self._check_deploy_task()

    def _build_resource(self):
        build_resource_task_params = {
            'target_resource_type': self.Parameters.resource_type,
            'arcadia_root_url': self.Parameters.arcadia_root_url,
            'arcadia_paths': self.Parameters.arcadia_paths,
            'folder_name': self.Parameters.folder_name,
        }
        if self.Parameters.version_file_path is not None:
            build_resource_task_params['version_file'] = self.Parameters.version_file_path

        build_resource_task = sdk2.Task[VideoVideohubDumpFilesFromArcadia.VideoVideohubDumpFilesFromArcadia.type](
            self,
            description='Dump confings from arcadia',
            **build_resource_task_params
        )
        build_resource_task.save().enqueue()
        self.Context.build_resource_task_id = build_resource_task.id
        self.Context.save()

        raise sdk2.WaitTask(
            [self.Context.build_resource_task_id],
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=True
        )

    def _release_task(self):
        self.server.release(
            task_id=self.Context.build_resource_task_id,
            type=ctt.ReleaseStatus.STABLE,
            subject=self.Parameters.release_title
        )

    def _is_changed_resource(self):
        build_resource_task = sdk2.Task.find(id=self.Context.build_resource_task_id).first()
        if build_resource_task is None:
            raise common.errors.TaskFailure("Can't find build resource task")

        new_resource = sdk2.Resource[self.Parameters.resource_type].find(task=build_resource_task).first()

        if new_resource is None:
            raise common.errors.TaskFailure("Can't find config resource")

        released_data = apihelpers.get_last_released_resource(self.Parameters.resource_type)
        if released_data and released_data.skynet_id == new_resource.skynet_id:
            return False
        return True

    def _run_dashboard_task(self):
        deploy_task_params = {
            'deployment_nanny_dashboard_name': self.Parameters.nanny_dashboard_name,
            'deployment_nanny_dashboard_recipe': self.Parameters.nanny_dashboard_recipe,
            'deployment_task_id': self.Context.build_resource_task_id,
            'services_from_recipe': self.Parameters.services_from_recipe,
            'deployment_release_status': self.Parameters.deployment_release_status,
            'vault_name': self.Parameters.vault_name,
            'vault_owner': self.Parameters.vault_owner,
            'deployment_nanny_bool_wait': True,
            'semaphore_name': self.Parameters.semaphore_name,
        }
        deploy_task = sdk2.Task[DeployNannyDashboard.DeployNannyDashboard.type](
            self,
            description='Run deployment.',
            **deploy_task_params
        )
        deploy_task.save().enqueue()
        self.Context.deploy_task_id = deploy_task.id
        self.Context.save()
        raise sdk2.WaitTask(
            [self.Context.deploy_task_id],
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=True
        )

    def _check_build_resource_task(self):
        if self.Context.build_resource_task_id is not NotExists:
            child = sdk2.Task[self.Context.build_resource_task_id]
            if child.status != ctt.Status.SUCCESS:
                raise common.errors.TaskFailure("Build resource task was finished with status {}".format(child.status))

    def _check_deploy_task(self):
        if self.Context.deploy_task_id is not NotExists:
            child = sdk2.Task[self.Context.deploy_task_id]
            if child.status != ctt.Status.SUCCESS:
                raise common.errors.TaskFailure("Deploy task was finished with status {}".format(child.status))
