from datetime import datetime
import logging
import os

from sandbox import common
from sandbox import sdk2

from sandbox.projects.common import utils
from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.grut.resources import GrutAdminTool


STAGES = ["testing", "stable"]


class GrutBackupDb(sdk2.Task):
    """ Task for performing GrUT database backup """

    class Requirements(sdk2.Task.Requirements):
        environments = [environments.PipEnvironment("yandex-yt")]

    class Parameters(sdk2.Task.Parameters):

        yt_token = sdk2.parameters.YavSecret(
            "Yav secret with YT token"
        )

        stage = sdk2.parameters.String(
            "Stage for backup",
            choices=[(s, s) for s in STAGES],
            default="stable",
            required=True
        )

        replica = sdk2.parameters.String(
            "Replica to make backup from",
            required=False
        )

        backup_cluster = sdk2.parameters.String(
            "Cluster to store backup",
            default="hahn",
            required=True
        )
        backup_dir = sdk2.parameters.String(
            "Directory to store backup",
            required=True
        )

        admin_tool_resource = sdk2.parameters.Resource(
            "Resource GrUT admin tool executable",
            resource_type=GrutAdminTool
        )

        backup_count = sdk2.parameters.Integer(
            "Count of backups to keep",
            default=7,
            required=True
        )

    def _clean_old_backups(self, stage_backup_dir):
        import yt.wrapper as yt

        yt.config["token"] = self.Parameters.yt_token.value()
        yt.config["proxy"]["url"] = self.Parameters.backup_cluster

        logging.info("Cleaning backups. Need to keep {} backups.".format(self.Parameters.backup_count))

        backups = sorted(yt.list(stage_backup_dir, absolute=False))
        backup_count_to_delete = max(0, len(backups) - self.Parameters.backup_count)

        logging.info("{} backups will be deleted.".format(backup_count_to_delete))

        for i in range(0, backup_count_to_delete):
            backup_path = yt.ypath_join(stage_backup_dir, backups[i])
            logging.info("Removing backup {}...".format(backup_path))
            yt.remove(backup_path, recursive=True)

        logging.info("Backup cleaning finished.")

    def _link_to_latest(self, stage_backup_dir, target_dir):
        import yt.wrapper as yt

        yt.config["token"] = self.Parameters.yt_token.value()
        yt.config["proxy"]["url"] = self.Parameters.backup_cluster

        latest_link = yt.ypath_join(stage_backup_dir, "latest")
        yt.link(target_dir, latest_link, force=True)
        logging.info("Set latest link to {}".format(target_dir))

    def on_execute(self):
        import yt.wrapper as yt

        if self.Parameters.admin_tool_resource:
            tool_path = str(sdk2.ResourceData(self.Parameters.admin_tool_resource).path)
        else:
            resource_id = utils.get_and_check_last_released_resource_id(GrutAdminTool)
            logging.info("Found last released admin tool: %s", str(resource_id))
            tool_path = str(sdk2.ResourceData(sdk2.Resource[resource_id]).path)

        env = os.environ.copy()
        env["YT_TOKEN"] = self.Parameters.yt_token.value()

        stage_backup_dir = yt.ypath_join(self.Parameters.backup_dir, self.Parameters.stage)
        target_dir = yt.ypath_join(stage_backup_dir, datetime.now().strftime("%Y-%m-%d %H:%M:%S"))

        cmd = [
            tool_path,
            "backup",
            "-S", self.Parameters.stage,
            "-T", self.Parameters.backup_cluster,
            "-D", target_dir
        ]
        if self.Parameters.replica:
            cmd.extend(["-R", self.Parameters.replica])

        logging.info("Start database backup with cmd: {}.".format(cmd))

        with sdk2.helpers.ProcessLog(self, logger="grut_admin") as logger:
            return_code = sp.Popen(cmd, stdout=logger.stdout, stderr=sp.STDOUT, env=env).wait()
            if return_code == 0:
                logging.info("Database backup finished successfully.")
            else:
                logging.error("Database backup failed. See attached logs.")
                raise common.errors.TaskFailure('Backup failed')

        self._link_to_latest(stage_backup_dir, target_dir)
        self._clean_old_backups(stage_backup_dir)
