import logging
import os

from sandbox import common
from sandbox import sdk2

import sandbox.sandboxsdk.environments as envs

from sandbox.projects.common import utils
from sandbox.projects.grut.resources import GrutBannerStatesDiffTool
from sandbox.sdk2.helpers import subprocess as sp


STAGES = ["testing", "stable"]


class GrutBannerStatesDiff(sdk2.Task):
    """ Task for building diff betweeen banner states in GrUT and in banner profiles """

    class Requirements(sdk2.Task.Requirements):
        environments = [envs.PipEnvironment("yandex-yt")]

    class Parameters(sdk2.Task.Parameters):

        tool_resource = sdk2.parameters.Resource(
            "Resource with GrUT banner states diff tool executable",
            resource_type=GrutBannerStatesDiffTool
        )

        yt_token = sdk2.parameters.YavSecret(
            "Yav secret with YT token",
            required=True
        )

        yt_cluster = sdk2.parameters.String(
            "Cluster with data",
            default="hahn",
            required=True
        )
        stage = sdk2.parameters.String(
            "Stage",
            choices=[(s, s) for s in STAGES],
            default="stable",
            required=True
        )
        backup_dir = sdk2.parameters.String(
            "Directory with grut backups",
            default="//home/grut/backups",
            required=True
        )
        caesar_table = sdk2.parameters.String(
            "Caesar table with banner profiles",
            required=True
        )
        output_table = sdk2.parameters.String(
            "Output table with banner ids",
            required=True
        )
        age_threshold = sdk2.parameters.String(
            "Skip banners created in the last `age_threshold` (10s, 10m, 10h)",
            default="0s"
        )

    def _get_last_backup_dir(self):
        import yt.wrapper as yt

        yt.config["token"] = self.Parameters.yt_token.value()
        yt.config["proxy"]["url"] = self.Parameters.yt_cluster

        stage_backup_dir = yt.ypath_join(self.Parameters.backup_dir, self.Parameters.stage)
        backups = sorted(yt.list(stage_backup_dir, absolute=True))
        last_backup = backups[-1]
        logging.info("Found last GrUT backup: %s", last_backup)
        return last_backup

    def on_execute(self):
        if self.Parameters.tool_resource:
            tool_path = str(sdk2.ResourceData(self.Parameters.tool_resource).path)
        else:
            resource_id = utils.get_and_check_last_released_resource_id(GrutBannerStatesDiffTool)
            logging.info("Found last released banner_states_diff tool: %s", str(resource_id))
            tool_path = str(sdk2.ResourceData(sdk2.Resource[resource_id]).path)

        env = os.environ.copy()
        env["YT_TOKEN"] = self.Parameters.yt_token.value()

        cmd = [
            tool_path,
            "--cluster", self.Parameters.yt_cluster,
            "--grut-backup-dir", self._get_last_backup_dir(),
            "--caesar-table", self.Parameters.caesar_table,
            "--output-table", self.Parameters.output_table,
            "--age-threshold", self.Parameters.age_threshold,
        ]

        with sdk2.helpers.ProcessLog(self, logger="banner_states_diff") as logger:
            return_code = sp.Popen(cmd, stdout=logger.stdout, stderr=sp.STDOUT, env=env).wait()
            if return_code == 0:
                logging.info("Banner states diff tool finished successfully.")
            else:
                logging.error("Banner states diff tool failed. See attached logs.")
                raise common.errors.TaskFailure('Banner states diff tool failed')
