import logging
import os

from sandbox import common
from sandbox import sdk2

import sandbox.sandboxsdk.environments as envs

from sandbox.projects.common import utils
from sandbox.projects.grut.resources import GrutBannerlandSyncTool
from sandbox.sdk2.helpers import subprocess as sp


STAGES = ["testing", "stable"]


class GrutBannerlandSync(sdk2.Task):
    """ Task for removing absent bannerland banners from GrUT """

    class Requirements(sdk2.Task.Requirements):
        environments = [envs.PipEnvironment("yandex-yt")]

    class Parameters(sdk2.Task.Parameters):

        tool_resource = sdk2.parameters.Resource(
            "Resource with GrUT to Bannerland sync tool executable",
            resource_type=GrutBannerlandSyncTool
        )

        yt_token = sdk2.parameters.YavSecret(
            "Yav secret with YT token",
            required=True
        )

        grut_token = sdk2.parameters.YavSecret(
            "Yav secret with GrUT token",
            required=True
        )

        yt_cluster = sdk2.parameters.String(
            "Cluster with data",
            default="hahn",
            required=True
        )
        stage = sdk2.parameters.String(
            "Stage",
            choices=[(s, s) for s in STAGES],
            default="stable",
            required=True
        )
        backup_dir = sdk2.parameters.String(
            "Directory with grut backups",
            default="//home/grut/backups",
            required=True
        )
        bannerland_table = sdk2.parameters.String(
            "Bannerland table with SelectionRank profiles",
            required=True
        )

        object_api_address = sdk2.parameters.String(
            "ObjectApi address",
            required=True
        )

        creation_lag = sdk2.parameters.Integer(
            "Lag of banner creation timestamp (hours)",
            default=2,
            required=True
        )

    def _get_last_backup_dir(self):
        import yt.wrapper as yt

        yt.config["token"] = self.Parameters.yt_token.value()
        yt.config["proxy"]["url"] = self.Parameters.yt_cluster

        stage_backup_dir = yt.ypath_join(self.Parameters.backup_dir, self.Parameters.stage)
        backups = sorted(yt.list(stage_backup_dir, absolute=True))
        last_backup = backups[-1]
        logging.info("Found last GrUT backup: %s", last_backup)
        return last_backup

    def on_execute(self):
        if self.Parameters.tool_resource:
            tool_path = str(sdk2.ResourceData(self.Parameters.tool_resource).path)
        else:
            resource_id = utils.get_and_check_last_released_resource_id(GrutBannerlandSyncTool)
            logging.info("Found last released bannerland_syn tool: %s", str(resource_id))
            tool_path = str(sdk2.ResourceData(sdk2.Resource[resource_id]).path)

        env = os.environ.copy()
        env["YT_TOKEN"] = self.Parameters.yt_token.value()
        env["GRUT_TOKEN"] = self.Parameters.grut_token.value()

        cmd = [
            tool_path,
            "--cluster", self.Parameters.yt_cluster,
            "--grut-backup-dir", self._get_last_backup_dir(),
            "--bannerland-table", self.Parameters.bannerland_table,
            "--object-api-address", self.Parameters.object_api_address,
            "--creation-lag", str(self.Parameters.creation_lag)
        ]

        with sdk2.helpers.ProcessLog(self, logger="bannerland_sync") as logger:
            return_code = sp.Popen(cmd, stdout=logger.stdout, stderr=sp.STDOUT, env=env).wait()
            if return_code == 0:
                logging.info("Bannerland sync finished successfully.")
            else:
                logging.error("Bannerland sync failed. See attached logs.")
                raise common.errors.TaskFailure('Bannerland sync failed')
