from datetime import datetime
import logging
import os

from sandbox import common
from sandbox import sdk2

import sandbox.sandboxsdk.environments as envs

from sandbox.projects.common import utils
from sandbox.projects.grut.resources import GrutFindInconsistentBannersTool
from sandbox.sdk2.helpers import subprocess as sp


STAGES = ["dev", "testing", "stable"]


class GrutFindInconsistentBanners(sdk2.Task):
    """ Task for finding banner candidates not synced with v2 """

    class Requirements(sdk2.Task.Requirements):
        environments = [envs.PipEnvironment("yandex-yt")]

    class Parameters(sdk2.Task.Parameters):
        tool_resource = sdk2.parameters.Resource(
            "Resource with tool executable",
            resource_type=GrutFindInconsistentBannersTool
        )

        yt_token = sdk2.parameters.YavSecret(
            "Yav secret with YT token",
            required=True
        )

        yt_cluster = sdk2.parameters.String(
            "Cluster with data",
            default="hahn",
            required=True
        )
        stage = sdk2.parameters.String(
            "Stage",
            choices=[(s, s) for s in STAGES],
            default="stable",
            required=True
        )
        backup_dir = sdk2.parameters.String(
            "Directory with grut backups",
            default="//home/grut/backups",
            required=True
        )
        output_dir = sdk2.parameters.String(
            "Output dir for tables",
            required=True
        )

    def _get_last_backup_dir(self):
        import yt.wrapper as yt

        yt.config["token"] = self.Parameters.yt_token.value()
        yt.config["proxy"]["url"] = self.Parameters.yt_cluster

        backup_dir_link = yt.ypath_join(self.Parameters.backup_dir, self.Parameters.stage, "latest")
        backup_dir_path = yt.get(yt.ypath_join(backup_dir_link, "@path"))
        logging.info("Found last GrUT backup: %s", backup_dir_path)
        return backup_dir_path

    def on_execute(self):
        import yt.wrapper as yt

        yt.config["token"] = self.Parameters.yt_token.value()
        yt.config["proxy"]["url"] = self.Parameters.yt_cluster

        if self.Parameters.tool_resource:
            tool_path = str(sdk2.ResourceData(self.Parameters.tool_resource).path)
        else:
            resource_id = utils.get_and_check_last_released_resource_id(GrutFindInconsistentBannersTool)
            logging.info("Found last released find_inconsistent_banners tool: %s", str(resource_id))
            tool_path = str(sdk2.ResourceData(sdk2.Resource[resource_id]).path)

        env = os.environ.copy()
        env["YT_TOKEN"] = self.Parameters.yt_token.value()

        output_table = yt.ypath_join(self.Parameters.output_dir, datetime.now().strftime("%Y-%m-%d %H:%M:%S"))

        cmd = [
            tool_path,
            "--cluster", self.Parameters.yt_cluster,
            "--grut-backup-dir", self._get_last_backup_dir(),
            "--output-table", output_table
        ]

        with sdk2.helpers.ProcessLog(self, logger="find_inconsistent_banners") as logger:
            return_code = sp.Popen(cmd, stdout=logger.stdout, stderr=sp.STDOUT, env=env).wait()
            if return_code == 0:
                logging.info("Find inconsistent banners tool finished successfully.")
            else:
                logging.error("Find inconsistent banners tool failed. See attached logs.")
                raise common.errors.TaskFailure('Find inconsistent banners tool failed')
