# -*- coding: utf-8 -*-
import logging
import os
import subprocess

from sandbox import common
from sandbox import sdk2

from sandbox.projects.common import vcs
from sandbox.sdk2.helpers import subprocess as sp


REVIEW_BRANCH = "generate-proto-scheme-md"

TOC_YAML_HEADER = """title: {title}
items:
  - name: Введение
    href: index.md
  - name: Схема данных
    items:
"""

ITEM_FORMAT = """      - name: Message {name}
        href: messages/{filename}
"""


class GrutProtoDocsGenerator(sdk2.Task):
    """
    task to generate protos objects schemes
    """

    class Parameters(sdk2.Task.Parameters):

        generator_arc_dir = sdk2.parameters.String(
            "directory with docs generator",
            required=True,
        )

        output_path = sdk2.parameters.String(
            "output docs path",
            required=True,
        )

        single_file = sdk2.parameters.Bool(
            "proto2doc output format",
            required=False,
            default=True,
        )

        expanded_output = sdk2.parameters.Bool(
            "Hide each message fields under the hood",
            required=False,
            default=False,
        )

    def _make_pr(self, arc, arc_path, author, branch, message, docs_dir):
        if not self.Parameters.single_file:
            arc.add(arc_path, os.path.join(docs_dir, "messages"))
        arc.commit(arc_path, message, all_changes=True)
        arc.push(arc_path, upstream="users/{}/{}".format(author, branch), force=True)
        try:
            arc.pr_create(arc_path, message=message, publish=True, auto=True)
        except vcs.arc.ArcCommandFailed as exc:
            # unfortunately sdk does not provide a better way of handling it
            if "already exists" not in exc.args[0]:
                raise

    def build(self, arc_path, target):
        ya = os.path.join(arc_path, "ya")
        printer_path = os.path.join(arc_path, target)
        subprocess.check_call([ya, "make", printer_path])
        return os.path.join(printer_path, "proto2doc")

    def get_printer_path(self, arc_path):
        return self.build(arc_path, self.Parameters.generator_arc_dir)

    def on_execute(self):
        arc = vcs.arc.Arc(arc_oauth_token=os.getenv("ARC_TOKEN"))
        docs_dir = os.path.dirname(self.Parameters.output_path)

        with arc.mount_path("", changeset="trunk", fetch_all=False) as arc_path:
            printer_path = self.get_printer_path(arc_path)
            logging.info("printer_path %s", printer_path)

            arc.checkout(arc_path, branch=REVIEW_BRANCH, create_branch=True, force=True)

            proto2docs_output_path = os.path.join(arc_path, self.Parameters.output_path)

            if not self.Parameters.single_file:
                protos_dir = os.path.join(arc_path, docs_dir, "messages")
                proto2docs_output_path = protos_dir
                if not os.path.exists(protos_dir):
                    os.makedirs(protos_dir)

            with sdk2.helpers.ProcessLog(self, logger="proto2doc") as logger:
                return_code = sp.Popen(
                    [
                        printer_path,
                        "-O" if self.Parameters.single_file else "-D", proto2docs_output_path,
                        "-F", "markdown",
                        "-C", "EXPANDED" if self.Parameters.expanded_output else "ENABLED",
                    ],
                    stdout=logger.stdout,
                    stderr=sp.STDOUT
                ).wait()
                if return_code == 0:
                    logging.info("Proto schemes are printed successfully.")

                    if not self.Parameters.single_file:
                        (_, _, filenames) = next(os.walk(protos_dir))

                        toc_title = None
                        toc_path = os.path.join(arc_path, docs_dir, "toc.yaml")

                        with open(toc_path, "r") as toc:
                            for line in toc.readlines():
                                if line.startswith("title:"):
                                    toc_title = line.split()[-1]
                                    break

                        if toc_title is None:
                            logging.error("Your docs should already have toc.yaml with defined title")
                            raise common.errors.TaskFailure("Undefined title for docs")

                        with open(toc_path, "w") as toc:
                            toc.write(TOC_YAML_HEADER.format(title=toc_title))
                            for f in filenames:
                                toc.write(ITEM_FORMAT.format(name=f.split(".")[0], filename=f))

                    self._make_pr(
                        arc,
                        arc_path,
                        self.author,
                        branch=REVIEW_BRANCH,
                        message="AUTO: Update documentation according to protobuf schema",
                        docs_dir=docs_dir,
                    )
                else:
                    logging.error("Proto schemes printer failed. See attached logs.")
                    raise common.errors.TaskFailure("Proto schemes printer failed")
