# -*- coding: utf-8 -*-
import datetime
import logging
import json

from sandbox import sdk2

from sandbox.projects.common import task_env
from sandbox.common.types import task as ctt

import sandbox_groups
import requests


class OutputQuotasResource(sdk2.Resource):
    pass


_TEMPLATE = """<!DOCTYPE html>
<html>
<head>
<title>Sandbox Quota Usage Report</title>
<style>
th {{
    text-align: left;
}}
</style>
</head>
<body>
    <table style="table-layout: fixed; width: 90%">
    <tr>
        <th>VERDICT</th>
        <th>RATIO</th>
        <th>USAGE</th>
        <th>QUOTA, mQP</th>
        <th>CORES</th>
        <th style="width: 400px">NAME</th>
        <th>ABC</th>
        <th>QUOTA</th>
        <th>TASKS</th>
        <th>PARENT</th>
    </tr>
    {table_body}
    </table>
</body>
</html>
"""

_LINE_FORMAT_HTML = """
<tr>
<td>{verdict}</td>
<td>{ratio:.5f}</td>
<td>{real}</td>
<td>{limit}</td>
<td>{cores}</td>
<td>{name}</td>
<td>{abc_link}</td>
<td>{quota_link}</td>
<td>{tasks_link}</td>
<td>{parent}</td>
</tr>"""

_TEMPLATE_ABC = """<!DOCTYPE html>
<html>
<head>
<title>Sandbox Quota Usage Report</title>
<style>
th {{
    text-align: left;
}}
</style>
</head>
<body>
    <table style="table-layout: fixed; width: 90%">
    <tr>
        <th style="width: 400px">NAME</th>
        <th>ABC</th>
    </tr>
    {table_body}
    </table>
</body>
</html>
"""

_LINE_FORMAT_HTML_ABC = """
<tr>
<td>{name}</td>
<td>{abc_link}</td>
</tr>"""


_ONE_HOST = 13500


class YtTokenYavSecret(sdk2.parameters.YavSecret):
    name = 'test_yav'
    description = 'Secret with YT store token'
    required = False


class CalculateSandboxQuotas(sdk2.Task):
    """
    Devoted to https://clubs.at.yandex-team.ru/sandbox/3417
    """

    class Requirements(task_env.TinyRequirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        min_quota = sdk2.parameters.Integer(
            "Minimal quota treshold",
            default_value=3000,
        )
        use_groups_dump = sdk2.parameters.Bool(
            "Use groups JSON dump instead of API",
            default_value=False,
        )
        get_list_groups_without_abc = sdk2.parameters.Bool(
            "Output: list groups without abc or maybe abc. Need abc_token",
            default_value=False,
        )
        abc_token_owner = sdk2.parameters.String("Abc token owner")
        abc_token_name = sdk2.parameters.String("Abc token name")
        name_filter = sdk2.parameters.String("Group name filter")
        test_yav = YtTokenYavSecret()

    def on_execute(self):
        output_resource = OutputQuotasResource(self, "Output file", "output_dir")

        # read groups data
        groups_dump = None
        if self.Parameters.use_groups_dump:
            groups_dump = json.loads(sandbox_groups.SANDBOX_GROUPS)
        else:
            groups_dump = self.server.group.read(limit=5000)
        logging.debug("GROUPS DUMP:\n\n\n%s\n\n\n", json.dumps(groups_dump, indent=4, sort_keys=True))
        groups = [group for group in groups_dump['items']]

        self._analyze(
            groups,
            output_resource,
            mode='ratio',
            sort_by_ratio=True,
            name_filter=self.Parameters.name_filter,
        )
        self._analyze(
            groups,
            output_resource,
            mode='usage',
            sort_by_ratio=False,
            name_filter=self.Parameters.name_filter,
        )
        if self.Parameters.get_list_groups_without_abc:
            self.abc_token = sdk2.Vault.data(self.Parameters.abc_token_owner, self.Parameters.abc_token_name)
            self._all_groups_without_abc(groups, output_resource,)

    def _all_groups_without_abc(self, groups, output_resource):
        out_lines = []
        out_lines.append('<!doctype html>\n<html><body>')
        for group in groups:
            name = group['name']
            abc = group.get('abc', '')
            possible_abc_names = [name]
            abc_link = '-'
            if not abc:
                if '_' in name:
                    possible_abc_names.append(name.replace('_', '-'))
                    possible_abc_names.append(name.replace('_', ''))
                logging.debug('name %s maybe_abc %s', name, possible_abc_names)
                headers = {
                    'Authorization': 'OAuth {abc_token}'.format(abc_token=self.abc_token)
                }
                for possible_abc in possible_abc_names:
                    response = requests.get(
                        "https://abc-back.yandex-team.ru/api/v4/services/?slug={abc}".format(abc=possible_abc),
                        headers=headers,
                    )
                    response = response.json()
                    if response.get('results'):
                        abc_link = (
                            '<a href="https://abc.yandex-team.ru/services/{abc}">{abc}</a>'.format(abc=possible_abc)
                        )
                        break
                out_line_html = _LINE_FORMAT_HTML_ABC.format(
                    name=name,
                    abc_link=abc_link,
                )
                out_lines.append(out_line_html)

        output_resource_data = sdk2.ResourceData(output_resource)
        output_resource_data.path.mkdir(0o755, parents=True, exist_ok=True)
        output_resource_data.path.joinpath("groups_without_abc_and_maybe_abc.html").write_bytes(
            _TEMPLATE_ABC.format(table_body='\n'.join(out_lines))
        )

    def _check_zero_usage(self, name):
        # override verdicts in some cases
        verdict = 'ZERO'

        recent_task = None
        try:
            now = datetime.datetime.utcnow()
            recent_task = sdk2.Task.find(
                owner=name,
                children=True,
                hidden=True,
                status=ctt.Status.SUCCESS,
                created='{}..{}'.format(
                    (now - datetime.timedelta(days=30)).isoformat(),
                    now.isoformat(),
                ),
            ).first()
        except Exception as e:
            logging.error('Group %s, error: %s', e)

        if recent_task:
            logging.info('TASK EXISTS: %s', recent_task.id)
            verdict = 'ALMOST_ZERO'
        return verdict

    def _analyze(self, groups, output_resource, mode, sort_by_ratio, name_filter=None):
        top = []
        out_lines = []
        out_lines.append('<!doctype html>\n<html><body>')

        total_limit = 0

        for group in groups:
            logging.debug('GROUP\n%s', json.dumps(group, indent=4))
            quota = group['quota']
            name = group['name']
            real = quota['consumption']['real']
            limit = quota.get('limit', 0)
            ratio = 1.0
            if limit is None:
                logging.debug('NONE LIMIT:\n%s', json.dumps(group, indent=4))
                limit = 0
            else:
                ratio = 1.0 - (1.0 * real / limit)

            total_limit += limit

            if limit > _ONE_HOST * 0.5 or ratio < 0.4:
                top.append((group, limit, ratio))

        if mode == 'ratio':
            self.set_info('Total limit (hosts): {}'.format(total_limit / _ONE_HOST))

        top_sorted = sorted(top, key=lambda x: (x[2 if sort_by_ratio else 1], x[1]), reverse=True)

        total_hosts = 0

        for group, limit, ratio in top_sorted:

            if name_filter and name_filter not in group['name']:
                continue

            quota = group['quota']
            name = group['name']
            abc = group.get('abc', '')
            hosts = int(limit / _ONE_HOST)
            real = quota['consumption']['real']

            low = False
            if limit >= _ONE_HOST and real * 2 < limit:
                low = True
            if real * 4 < limit:
                low = True

            zero = real < 100
            overcommit = real > limit

            verdict = None

            if overcommit:
                verdict = 'OVERCOMMIT'

            if low:
                verdict = 'NOT_EFFECTIVE'

            if zero:
                verdict = self._check_zero_usage(name)

            if not abc:
                verdict = 'ABC_NOT_SET'

            if verdict:
                quota_link = (
                    'https://grafana.yandex-team.ru/d/000015873/sandbox-quota-consumption-production?'
                    'orgId=1&var-owner={name}&from=now-90d&to=now'.format(
                        name=name,
                    )
                )
                quota_link = '<a href="{}">Quota</a>'.format(quota_link)
                abc_link = (
                    '<a href="https://abc.yandex-team.ru/services/{abc}">{abc}</a>'.format(abc=abc)
                    if abc else '-'
                )
                tasks_link = (
                    '<a href="https://sandbox.yandex-team.ru/tasks?'
                    'children=true&hidden=true&limit=10&created=14_days&owner={}">Tasks</a>'
                ).format(name)

                parent = group.get('parent') or '-'
                out_line_html = _LINE_FORMAT_HTML.format(
                    verdict=verdict,
                    ratio=1.0 * real / limit,
                    real=real,
                    limit=limit,
                    cores=int(limit * 32 / _ONE_HOST),
                    name=name,
                    parent=parent,
                    abc_link=abc_link,
                    quota_link=quota_link,
                    tasks_link=tasks_link,
                )
                out_lines.append(out_line_html)

            total_hosts += hosts

        logging.info('%s TOTAL HOSTS', total_hosts)

        output_resource_data = sdk2.ResourceData(output_resource)
        output_resource_data.path.mkdir(0o755, parents=True, exist_ok=True)
        output_resource_data.path.joinpath(
            "quota_analysis_{mode}.html".format(mode=mode)
        ).write_bytes(_TEMPLATE.format(table_body='\n'.join(out_lines)))
