# -*- coding: utf-8 -*-

import logging
import time

from sandbox import sdk2
from sandbox import common
from sandbox.common.types.task import ReleaseStatus
from sandbox.projects.common import decorators
from sandbox.projects.common import requests_wrapper

from resource_types import YamdAcceptanceBegemotResource
from context_extractors import (HealthSaasRequestContext, TemplateRouterResponseContext, JsonFromText)
from context_validators import SaasRequestValidator


REQUEST_URL = 'http://hamster.yandex.ru/health/search'
REQUEST_TEMPLATE = (
    '{beta_url}?text={search_text}'
    '&dump=eventlog&dump_source_request=HEALTH_SAAS&dump_source_response=TEMPLATE_ROUTER'
)


class YamdAcceptanceBegemotGraphTask(sdk2.Task):
    class Parameters(sdk2.Parameters):
        beta_url = sdk2.parameters.String('Beta url', default=REQUEST_URL, required=True)

        json_basket_from_metriks = sdk2.parameters.Resource(
            label='json queries basket from metriks',
            required=False,
            resource_type=YamdAcceptanceBegemotResource
        )

    def on_execute(self):
        total_success = 0
        saas_extractor = HealthSaasRequestContext()
        router_extractor = TemplateRouterResponseContext()

        all_requests = self._get_search_texts()
        total_requests_quantity = len(all_requests)
        logging.info('Total requests quantity: {}'.format(total_requests_quantity))
        for i, search_text in enumerate(all_requests):
            url = self._get_url(search_text)
            html = _requests_get_r(url).text
            try:
                saas_json = saas_extractor.get_context_json_as_dict(html)
                router_json = router_extractor.get_context_json_as_dict(html)
                saas_validator = SaasRequestValidator(search_text)
                saas_validator.check_equality(expected_object=router_json, actual_object=saas_json)
                logging.info('Text={}. OK. Request #{}'.format(search_text, i))
                total_success += 1
            except Exception as e:
                logging.error(
                    'Unexpected context for search text="{search_text}", error details={error_details}. Full url={url}'
                    .format(
                        search_text=search_text,
                        error_details=str(e),
                        url=url
                    )
                )

        success = total_requests_quantity > 0 and total_success == total_requests_quantity
        if not success:
            self._raise_exception(
                'Test failed. Total unsuccess tests: {total_unsuccess}'.format(
                    total_unsuccess=(total_requests_quantity - total_success)
                )
            )
        else:
            logging.info('Test successfully passed!')

    def _get_search_texts(self):
        resource = self.Parameters.json_basket_from_metriks
        if not resource:
            logging.info('Parameter resource with queries basket wasnt set.')
            logging.info('Try find last released YAMD_ACCEPTANCE_BEGEMOT_RESOURCE resource')

            released = ReleaseStatus.STABLE
            resource = YamdAcceptanceBegemotResource.find(
                attrs={"released": released}
            ).first()

        if not resource:
            raise Exception('Released YAMD_ACCEPTANCE_BEGEMOT_RESOURCE resource not found.')
        resource_data = sdk2.ResourceData(resource)
        basket = resource_data.path.read_text()
        return [record['text'] for record in JsonFromText.get_json_from_string(basket)]

    def _get_url(self, search_text):
        return (
            REQUEST_TEMPLATE.format(
                beta_url=self.Parameters.beta_url,
                search_text=search_text
            )
        )

    def _raise_exception(self, message):
        logging.error(message)
        raise common.errors.TaskError(message)


@decorators.retries(max_tries=3, delay=3, backoff=2)
def _requests_get_r(*args, **kwargs):
    """
    Copied from websearch/TestXmlSearch.
    Same as `requests_wrapper.get_r` but with custom retry policy
    and additional checking of empty responses
    (less backoff and delay)
    """
    raise_for_empty_response = kwargs.pop('raise_for_empty_response', False)
    r = requests_wrapper.get(*args, **kwargs)

    if r.status_code == 500:
        # handle hamster overload. Sample log: https://paste.yandex-team.ru/700720
        logging.info("Got 500 error, maybe hamster is overloaded, sleeping for 120s...")
        time.sleep(120)

    r.raise_for_status()

    if raise_for_empty_response and not r.text:
        raise Exception("Empty response received. ")

    return r
