# -*- coding: utf-8 -*-
import re
import json


class JsonFromText(object):
    @staticmethod
    def get_json_from_string(raw_json):
        cleared_json = JsonFromText.refine_string(raw_json)
        if not cleared_json:
            return {}

        try:
            actual_context = json.loads(cleared_json)
        except Exception as e:
            raise Exception('Cant load json={}. Exception: {}'.format(cleared_json, e))
        return actual_context

    @staticmethod
    def refine_string(string):
        if string:
            if type(string) is unicode:
                string = string.encode('utf-8')
            return string.decode('string_escape').strip()
        return ''


class ContextFromResponse(object):
    def __init__(self, context_type):
        self.context_type = context_type

    def get_context_json_as_dict(self, html_response):
        raw_json = self._get_context_from_eventlog(html_response)
        actual_context = self._get_json_from_string(raw_json)

        if 'answers' not in actual_context:
            raise Exception('No "answers" in context, for source="{}"'.format(self._get_source_name()))

        type_context = {}
        for type_response in actual_context['answers']:
            if type_response['type'] == self.context_type:
                type_context = type_response
                break

        if not type_context:
            raise Exception('No type="{}" from source="{}"'.format(self.context_type, self._get_source_name()))

        if 'binary' not in type_context:
            raise Exception(
                'Not found "binary" field in type={}, from source="{}". Context={}'.format(
                    self.context_type,
                    self._get_source_name(),
                    type_context
                )
            )
        return type_context['binary']

    def _get_json_from_string(self, raw_json):
        result = JsonFromText.get_json_from_string(raw_json)
        if not result:
            raise Exception('Empty acutal string response for source = "{}"'.format(self._get_source_name()))

        return result

    def _get_source_name(self):
        raise NotImplementedError

    def _get_eventlog_type(self):
        raise NotImplementedError

    def _get_context_from_eventlog(self, html):
        reg = re.compile(
            '{eventlog_type}\\\\t{source}(.*?)",\s+"\d+'.format(
                source=self._get_source_name(),
                eventlog_type=self._get_eventlog_type()
            ),
            re.DOTALL
        )
        m = reg.search(html)
        if m:
            return m.group(1)
        return ''


class HealthSaasRequestContext(ContextFromResponse):
    def __init__(self, context_type='saas_input'):
        super(HealthSaasRequestContext, self).__init__(context_type)

    def _get_eventlog_type(self):
        return 'TSourceRequest'

    def _get_source_name(self):
        return 'HEALTH_SAAS'


class TemplateRouterResponseContext(ContextFromResponse):
    def __init__(self, context_type='flags'):
        super(TemplateRouterResponseContext, self).__init__(context_type)

    def get_context_json_as_dict(self, html_response):
        flags_context = super(TemplateRouterResponseContext, self).get_context_json_as_dict(html_response)
        if 'all' not in flags_context:
            raise Exception('No "all" field in "flags" context')

        if 'health_settings' not in flags_context['all']:
            raise Exception('No "health_settings" field in "flags->all" context')

        return self._get_json_from_string(flags_context['all']['health_settings'])

    def _get_eventlog_type(self):
        return 'TSourceResponse'

    def _get_source_name(self):
        return 'TEMPLATE_ROUTER'
