# -*- coding: utf-8 -*-


class BaseValidator(object):
    def check_equality(self, expected_object, actual_object):
        not_equal_fields = []
        exception_msg = ''
        for field in self._get_compare_fields():
            if self._should_validate_in_child_class(field):
                try:
                    self._validate_in_child_class(expected_object, actual_object, field)
                except Exception as e:
                    exception_msg += '{}, '.format(str(e))
                continue

            if type(expected_object[field]) is list:
                if not self._validate_array_equality(expected_object[field], actual_object[field]):
                    not_equal_fields.append(field)

            else:
                if expected_object[field] != actual_object[field]:
                    not_equal_fields.append(field)

        if not_equal_fields:
            raise Exception(
                'Fields are not equal, details: {}, {}'.format(
                    self._get_not_equal_fields_info(expected_object, actual_object, not_equal_fields),
                    exception_msg
                )
            )

    def _should_validate_in_child_class(self, field):
        return False

    def _validate_in_child_class(self, expected_object, actual_object, field):
        raise NotImplementedError

    def _get_compare_fields(self):
        raise NotImplementedError

    def _validate_array_equality(self, larray, rarray):
        return set(larray) == set(rarray)

    def _get_not_equal_fields_info(self, expected_object, actual_object, not_equal_fields):
        result = ''
        for field in not_equal_fields:
            result += 'Field "{}", expected="{}", actual="{}"'.format(
                field,
                expected_object[field],
                actual_object[field]
            )
        return result


class SaasRequestValidator(BaseValidator):
    def __init__(self, search_text):
        self.search_text = search_text

    def _should_validate_in_child_class(self, field):
        return field in [
            'msp',
            'text',
            'user_request',
            'use_qtree',
            'qtree',
            'relev',
            'numdoc',
            'ag0'
        ]

    def _validate_in_child_class(self, template_router_response, health_saas_request, field):
        if field == 'msp':
            return health_saas_request[field] == 'no'

        if field == 'text':
            return len(health_saas_request[field]) > 0

        if field == 'user_request':
            return self.search_text in health_saas_request[field]

        if field == 'use_qtree' or field == 'qtree':
            if template_router_response['use_qtree']:
                return len(health_saas_request['qtree']) > 0
            return 'qtree' not in health_saas_request

        if field == 'ag0':
            if template_router_response[field]:
                return self._validate_array_equality(template_router_response[field], health_saas_request[field])
            return field not in health_saas_request

        if field == 'relev':
            if template_router_response[field]:
                return len(health_saas_request[field]) == 1 and health_saas_request[field][0] == template_router_response[field]
            return len(health_saas_request[field]) == 0

        if field == 'numdoc':
            return str(template_router_response[field]) == str(health_saas_request[field])

    def _get_compare_fields(self):
        return [
            'msp',
            'gta',
            'g',
            'service',
            'template',
            'kps',
            'text',
            'qtree',
            'use_qtree',
            'how',
            'relev',
            'user_request',
            'pron',
            'numdoc',
            'ag0'
        ]
