# -*- coding: utf-8 -*-
"""
Базовый класс задачки generate_yamd_database.
Используется sandbox'ым python'ом
для получения метаинформации о задаче (а именно):
- параметры
- тип задачи

Помимо этого
- содержит имплементацию server-side методов,
- использует имплементацию метода execute из
GenerateYamdDatabaseTaskImplementation
"""

import datetime
import logging

# Ресурсы задачи
from resource_types import YamdDatabaseResource  # noqa
from sandbox.projects.resource_types import OTHER_RESOURCE
from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types.task import ReleaseStatus
from sandbox.projects.common.nanny import nanny

default_release_subscribers = ['aleksey-titov', 'vladvolkov', 'denvr']


class BinaryType(ReleaseStatus):
    NONE = 'None'


class GenerateYamdDatabaseTask(sdk2.Task, nanny.ReleaseToNannyTask2):
    class Parameters(sdk2.Parameters):
        release_subscribers = sdk2.parameters.List(
            label='Release subscribers',
            default=default_release_subscribers,
            value_type=sdk2.parameters.Staff
        )

        with sdk2.parameters.RadioGroup(
            label="Release type of binary executor resource",
            group="Binary task executor"
        ) as binary_type:
            binary_type.values[BinaryType.STABLE] = binary_type.Value(BinaryType.STABLE)
            binary_type.values[BinaryType.TESTING] = binary_type.Value(BinaryType.TESTING)
            binary_type.values[BinaryType.NONE] = binary_type.Value(BinaryType.NONE, default=True)

        force = sdk2.parameters.Bool(
            label='Force (disable tests)?',
            default=False,
        )

        with sdk2.parameters.Group(
            label='Tests dependencies'
        ) as tests_dependencies:
            sqldiff_binary = sdk2.parameters.Resource(
                label='Sqldiff binary resource to compare databses',
                resource_type=OTHER_RESOURCE,
                required=False
            )

            old_db_resource = sdk2.parameters.Resource(
                label='Last stable database resource to with compare new database. By default - latest stable',
                resource_type=YamdDatabaseResource,
                required=False
            )

        with sdk2.parameters.Group(
            label='Tests parameters',
        ) as tests_parameters:
            # todo: figure defaults here
            max_changes_percent = sdk2.parameters.Float(
                'Max CHANGES (~) percent between releases allowed [0, 1]', default=0.3
            )
            max_inserts_percent = sdk2.parameters.Float(
                'Max INSERTS (+) percent between releases allowed [0, 1]', default=0.3
            )
            max_deletes_percent = sdk2.parameters.Float(
                'Max DELETES (-) percent between releases allowed [0, 1]', default=0.2
            )
            max_products_ids_without_rospharm_percent = sdk2.parameters.Float(
                'Max products without rospharm info percent allowed [0, 1]', default=0.4
            )

    # public
    def on_save(self):
        self._setup_binary_requirement()
        self._setup_old_database_requirement()
        self._setup_sqldiff_binary_requirement()

    def on_execute(self):
        self.validate_parameters()
        self.execute()

    def validate_parameters(self):
        self._validate_binary_executor_resource()

        if not self.Parameters.force:
            # we need some resources in this task.
            self._validate_old_db_resource()
            self._validate_sqldiff_binary_resource()

    def execute(self):
        import implementation
        implementation.GenerateYamdDatabaseTaskImplementation.on_execute(self)

    def on_success(self, prev_status):
        sdk2.Task.on_success(self, prev_status)
        self.on_release(self.release_parameters)

    def on_release(self, additional_parameters):
        logging.info('Auto-releasing task after success')
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)
        self.mark_released_resources(additional_parameters["release_status"], ttl=30)

    @property
    def release_parameters(self):
        if self.Parameters.binary_type == BinaryType.STABLE:
            status = ReleaseStatus.STABLE
        elif self.Parameters.binary_type == BinaryType.TESTING:
            status = ReleaseStatus.TESTING
        else:
            status = ReleaseStatus.UNSTABLE

        now = datetime.datetime.now().isoformat()
        release_subject = (
            '[{}] New auto generated Yamd database release '
            'from sandbox task {}'
        ).format(now, self.id)

        release_parameters = dict(
            releaser=self.author,
            release_status=status,
            release_subject=release_subject,
            email_notifications=dict(to=self.Parameters.release_subscribers, cc=[]),
            release_comments=release_subject,
        )

        return release_parameters

    # private
    def _setup_binary_requirement(self):
        if self.Parameters.binary_type != BinaryType.NONE:
            self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
                attrs={
                    "task_type": self.type.name,
                    "released": self.Parameters.binary_type
                }
            ).first()

    def _setup_sqldiff_binary_requirement(self):
        if not self.Parameters.sqldiff_binary:
            self.Parameters.sqldiff_binary = sdk2.Resource.find(
                attrs={"kind": 'sqldiff_binary'}
            ).first()

    def _setup_old_database_requirement(self):
        if not self.Parameters.old_db_resource:
            released = ReleaseStatus.STABLE  # todo: maybe choice based on task.Parameters.binary_type
            self.Parameters.old_db_resource = YamdDatabaseResource.find(
                attrs={"released": released}
            ).first()

    def _validate_sqldiff_binary_resource(self):
        if not self.Parameters.sqldiff_binary:
            error_message = (
                "Resource with kind = sqldiff_binary not found"
            )
            raise errors.TaskFailure(error_message)

    def _validate_old_db_resource(self):
        if not self.Parameters.old_db_resource:
            released = ReleaseStatus.STABLE  # todo: maybe choice based on task.Parameters.binary_type
            error_message = (
                'Old database with attr released={} not found. '
                'You can run task with "force" option this time to '
                'create similar database resource for the next run.'
            ).format(released)
            raise errors.TaskFailure(error_message)

    def _validate_binary_executor_resource(self):
        if not self.Requirements.tasks_resource:
            error_message = (
                "SandboxTasksBinary resource "
                "for task type {} with attribute released=={} not found"
            ).format(self.type.name, self.Parameters.binary_type)
            raise errors.TaskFailure(error_message)
