# -*- coding: utf-8 -*-
"""
Реализация задачи generate_yamd_database
Большая часть рабочего кода находится в health.yamd.health_import
"""

import logging

import health.yamd.health_import.configs.app_config as import_config
import health.yamd.health_import.data_tests as import_data_tests
import health.yamd.health_import.data_tests.tests_environment as import_tests_environment
import health.yamd.health_import.main as import_main
import resource_types
from sandbox import common
from sandbox import sdk2
from sandbox.common import errors


class SandboxVaultSecret(object):
    def __init__(self, owner, secret_name):
        self.owner = owner
        self.secret_name = secret_name

    def get_value(self):
        try:
            return sdk2.Vault.data(self.owner, self.secret_name)
        except common.errors.VaultError:
            error_message = "Cannot get required secret {}!".format(self.secret_name)
            raise common.errors.TaskFailure(error_message)


class GenerateYamdDatabaseTaskImplementation(object):
    @classmethod
    def on_execute(cls, task):
        # type: (cls, 'GenerateYamdDatabaseTask') -> None
        cls.__preset_config(task)
        new_db_resource = cls.__generate_database(task)

        if not task.Parameters.force:
            tests_env = cls.__prepare_environment_for_tests(task, new_db_resource)
            cls.__try_test_database(task, tests_env)

    @classmethod
    def __preset_config(cls, task):
        # type: (cls, 'GenerateYamdDatabaseTask') -> None
        pgmed_ro_constring = SandboxVaultSecret(task.owner, 'YA_HEALTH_PG_CONNECTION_STRING').get_value()
        yt_token = SandboxVaultSecret(task.owner, 'YA_HEALTH_YT_TOKEN').get_value()

        import_config.AppConfig.PG_CONNECTION_STRING = pgmed_ro_constring
        import_config.AppConfig.YT_TOKEN = yt_token

    @classmethod
    def __generate_database(cls, task):
        # type: (cls, 'GenerateYamdDatabaseTask') -> str
        filename = 'db.sqlite'
        db_resource = resource_types.YamdDatabaseResource(
            task,
            'Yandex Health YAMD sqlite database',
            path=filename
        )
        db_resource_path = str(db_resource.path)

        import_main.generate_yamd_database_in_file(db_resource_path)
        return db_resource_path

    @classmethod
    def __prepare_environment_for_tests(cls, task, new_db_resource_path):
        # type: (cls, 'GenerateYamdDatabaseTask', str) -> 'import_tests_environment.TestsEnvironment'
        old_db_resource_path = cls._download_resource(task.Parameters.old_db_resource)
        sqldiff_binary_path = cls._download_resource(task.Parameters.sqldiff_binary)
        tests_env = import_tests_environment.TestsEnvironment(
            old_db_resource_path,
            new_db_resource_path,
            sqldiff_binary_path,
            task.Parameters.max_changes_percent,
            task.Parameters.max_inserts_percent,
            task.Parameters.max_deletes_percent,
            task.Parameters.max_products_ids_without_rospharm_percent,
        )
        return tests_env

    @classmethod
    def __try_test_database(cls, task, tests_env):
        # type: (cls, 'GenerateYamdDatabaseTask', 'import_tests_environment.TestsEnvironment') -> None
        logging.info('Testing new database in comparison with older database')
        try:
            cls.__test_database(task, tests_env)
        except Exception as ex:
            error_message = 'Database tests were failed with error: {}'.format(ex)
            logging.exception(error_message)
            raise errors.TaskFailure(error_message)
        else:
            logging.info('Tests successfully passed')

    @classmethod
    def __test_database(cls, task, tests_env):
        import_data_tests.test_new_database(tests_env)

    @staticmethod
    def _download_resource(resource):
        logging.info('Downloading resource {}'.format(resource))
        resource_data = sdk2.ResourceData(resource)
        resource_data_path = str(resource_data.path)
        logging.info('Resource {} downloaded successfully to {} - OK'.format(resource, resource_data_path))
        return resource_data_path
