# -*- coding: utf-8 -*-
import json
import requests
import sandbox.common.types.misc as ctm
from sandbox import sdk2
import logging
from sandbox.common.errors import TaskError
from sandbox.sandboxsdk.environments import PipEnvironment

logger = logging.getLogger(__name__)


class HelpdeskSyncFromJsonToYt(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 1024
        cores = 4
        dns = ctm.DnsType.DNS64

        class Caches(sdk2.Requirements.Caches):
            pass

        environments = [
            PipEnvironment('yandex-yt')
        ]

    class Parameters(sdk2.Task.Parameters):

        DEFAULT_SCHEMA = [
            {"name": "id", "type": "string", "sort_order": "ascending"},
            {"name": "param1", "type": "string"},
            {"name": "param2", "type": "string"},
            {"name": "param3", "type": "string"}
        ]

        tokens = sdk2.parameters.YavSecret(
            "Yav secret with deploy tokens"
        )
        schema = sdk2.parameters.JSON(
            "Schema for new YT table", required=True, default=DEFAULT_SCHEMA
        )
        is_dynamic = sdk2.parameters.Bool("Check if need dynamic table", default_value=True)
        url = sdk2.parameters.Url(
            "Url with json response", required=True
        )
        yt_path = sdk2.parameters.Url(
            "Path in Yt hahn starts with //home/helpdesk/", required=True
        )

    def __create_table(self):
        logger.info('Creating YT Table')
        self.yt_client.create("table", self.yt_table, attributes={
            "schema": self.schema, "dynamic": self.Parameters.is_dynamic}, force=True)

        if self.Parameters.is_dynamic:
            self.yt_client.mount_table(self.yt_table)

    def write_data_to_yt(self, data):
        import yt.wrapper as yt
        self.check_or_create_table()

        if self.Parameters.is_dynamic:
            self.yt_client.mount_table(self.yt_table)
            data_len = int(len(data) / 100000) + 1

            for num in range(data_len):
                start = num * 100000
                end = (num + 1) * 100000
                self.yt_client.insert_rows(self.yt_table, data[start:end],
                                           update=True, format=yt.JsonFormat(attributes={"encode_utf8": False},
                                                                             encoding='utf-8'))
        else:
            table = yt.TablePath(name=self.yt_table, append=self.static_append)
            self.yt_client.write_table(table, data,
                                       format=yt.JsonFormat(attributes={"encode_utf8": False},
                                                            encoding='utf-8'))

    def check_or_create_table(self):
        if not self.yt_client.exists(self.yt_table) or self.recreate:
            logger.info(
                'Cannot find table with path {} creating it...'.format(self.yt_table))
            self.__create_table()

    def _fetch_data(self):
        headers = {
            "Authorization": 'OAuth {}'.format(self.Parameters.tokens.data()['toolsoauth'])
        }
        result = []
        results = requests.get(
            self.Parameters.url,
            headers=headers).json()
        approved_keys = [x['name'] for x in self.schema]
        for record in results:
            result.append(
                {key: record[key] for key in record if key in approved_keys})

        return result

    def make_hahn_client(self):
        import yt.wrapper as yt
        client = yt.client.YtClient(proxy=None, token=None, config=None)
        client.config["proxy"]["url"] = "hahn"
        client.config["token"] = self.Parameters.tokens.data()['yttoken']
        return client

    def on_execute(self):
        if self.Parameters.schema == self.Parameters.DEFAULT_SCHEMA:
            raise TaskError("Please specify schema")
        self.yt_table = '//home/helpdesk/{}'.format(self.Parameters.yt_path)
        self.recreate = False
        self.yt_client = self.make_hahn_client()
        self.static_append = True
        self.schema = json.loads(json.dumps(self.Parameters.schema))
        data = self._fetch_data()
        self.check_or_create_table()
        print("Start upload to YT")
        self.write_data_to_yt(data)
