from sandbox import sdk2

from sandbox.common import errors
from sandbox.common.types.task import Status

from sandbox.projects.common import link_builder as lb
from sandbox.projects.common.build.YaMake2 import YaMake2

from sandbox.projects.hollywood.common import resources
from sandbox.projects.hollywood.RunAnalyticsInfoTests import RunAnalyticsInfoTests


class BuildAndRunAnalyticsInfoTests(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 5 * 60  # This task doesn't do anything itself, 5 minutes is enough

        vins_url = sdk2.parameters.String(
            "Vins url",
            required=True,
        )

        experiments = sdk2.parameters.List(
            "List of experiments",
            default=[]
        )

        no_exec = sdk2.parameters.Bool(
            "Do not start the graph",
            default=False
        )

        with sdk2.parameters.Output:
            analytics_info_tests_binary = sdk2.parameters.Resource(
                "Analytics Info tests binary",
                resource_type=resources.AnalyticsInfoTestsBinary
            )

    def on_execute(self):
        with self.memoize_stage.build_analytics_info_tests:
            self.Context.build_task_id = YaMake2(
                self,
                description='Build Analytics Info tests',
                use_aapi_fuse=True,
                aapi_fallback=True,
                arts="alice/acceptance/runners/run",
                targets="alice/acceptance/runners",
                result_rt="ANALYTICS_INFO_TESTS_BINARY"
            ).enqueue().id

            self.set_info("build_task_id: {}".format(self.Context.build_task_id))
            raise sdk2.WaitTask(self.Context.build_task_id, Status.Group.FINISH | Status.Group.BREAK)

        with self.memoize_stage.check_build:
            build_task = sdk2.Task[self.Context.build_task_id]
            if build_task.status not in Status.Group.SUCCEED:
                self.set_info(
                    'Failed to build Analytics Info tests: {}'.format(lb.task_link(self.Context.build_task_id)),
                    do_escape=False
                )
                raise errors.TaskFailure('Failed to build')
            self.set_info(
                'Build Analytics Info tests succeeded: {}'.format(lb.task_link(self.Context.build_task_id)),
                do_escape=False
            )
            self.Parameters.analytics_info_tests_binary = resources.AnalyticsInfoTestsBinary.find(task=build_task).first().id

        with self.memoize_stage.run_analytics_info_tests:
            self.Context.run_task_id = RunAnalyticsInfoTests(
                self,
                vins_url=self.Parameters.vins_url,
                experiments=self.Parameters.experiments,
                analytics_info_tests_binary=self.Parameters.analytics_info_tests_binary,
                no_exec=self.Parameters.no_exec,
            ).enqueue().id
            self.set_info(
                "Running Analytics Info tests: {}".format(lb.task_link(self.Context.run_task_id)),
                do_escape=False
            )
            raise sdk2.WaitTask(self.Context.run_task_id, Status.Group.FINISH | Status.Group.BREAK)

        with self.memoize_stage.check_run:
            run_task = sdk2.Task[self.Context.run_task_id]
            if run_task.status not in Status.Group.SUCCEED:
                self.set_info(
                    'Failed to run Analytics Info tests: {}'.format(lb.task_link(self.Context.run_task_id)),
                    do_escape=False
                )
                raise errors.TaskFailure('Failed to run')
            self.set_info(
                'Analytics Info tests succeeded: {}'.format(lb.task_link(self.Context.run_task_id)),
                do_escape=False
            )
