from sandbox import sdk2

from sandbox.common import errors
from sandbox.common.types.task import Status

from sandbox.projects.common import link_builder as lb
from sandbox.projects.common.build.YaMake2 import YaMake2

from sandbox.projects.hollywood.common import resources
from sandbox.projects.hollywood.RunMarkerTests import RunMarkerTests


class BuildAndRunMarkerTests(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 5 * 60  # This task doesn't do anything itself, 5 minutes is enough

        vins_url = sdk2.parameters.String(
            "Vins url",
            required=True,
        )

        experiments = sdk2.parameters.List(
            "List of experiments",
            default=[]
        )

        no_exec = sdk2.parameters.Bool(
            "Do not start the graph",
            default=False
        )

        use_soy_testing = sdk2.parameters.Bool(
            "Use SoY testing contour. SCRAPEROVERYT-234",
            default=False
        )

        wait_nirvana = sdk2.parameters.Bool(
            "Wait nirvana graph to execute",
            default=False
        )

        with sdk2.parameters.Output:
            marker_tests_binary = sdk2.parameters.Resource("Marker tests binary", resource_type=resources.MarkerTestsBinary)

    def on_execute(self):
        with self.memoize_stage.build_marker_tests:
            self.Context.build_task_id = YaMake2(
                self,
                description='Build Marker tests',
                use_aapi_fuse=True,
                aapi_fallback=True,
                arts="alice/acceptance/cli/marker_tests/bin/marker_tests",
                targets="alice/acceptance/cli/marker_tests/bin",
                result_rt="MARKER_TESTS_BINARY"
            ).enqueue().id

            self.set_info("build_task_id: {}".format(self.Context.build_task_id))
            raise sdk2.WaitTask(self.Context.build_task_id, Status.Group.FINISH | Status.Group.BREAK)

        with self.memoize_stage.check_build:
            build_task = sdk2.Task[self.Context.build_task_id]
            if build_task.status not in Status.Group.SUCCEED:
                self.set_info(
                    'Failed to build Marker tests: {}'.format(lb.task_link(self.Context.build_task_id)),
                    do_escape=False
                )
                raise errors.TaskFailure('Failed to build')
            self.set_info(
                'Build Marker tests succeeded: {}'.format(lb.task_link(self.Context.build_task_id)),
                do_escape=False
            )
            self.Parameters.marker_tests_binary = resources.MarkerTestsBinary.find(task=build_task).first().id

        with self.memoize_stage.run_marker_tests:
            additional_args = {}
            if self.Parameters.use_soy_testing:
                additional_args['kill_timeout'] = 60 * 60
            self.Context.run_task_id = RunMarkerTests(
                self,
                vins_url=self.Parameters.vins_url,
                experiments=self.Parameters.experiments,
                marker_tests_binary=self.Parameters.marker_tests_binary,
                no_exec=self.Parameters.no_exec,
                use_soy_testing=self.Parameters.use_soy_testing,
                wait_nirvana=self.Parameters.wait_nirvana,
                **additional_args
            ).enqueue().id
            self.set_info(
                "Running Marker tests: {}".format(lb.task_link(self.Context.run_task_id)),
                do_escape=False
            )
            raise sdk2.WaitTask(self.Context.run_task_id, Status.Group.FINISH | Status.Group.BREAK)

        with self.memoize_stage.check_run:
            run_task = sdk2.Task[self.Context.run_task_id]
            if run_task.status not in Status.Group.SUCCEED:
                self.set_info(
                    'Failed to run Marker tests: {}'.format(lb.task_link(self.Context.run_task_id)),
                    do_escape=False
                )
                raise errors.TaskFailure('Failed to run')
            self.set_info(
                'Marker tests succeeded: {}'.format(lb.task_link(self.Context.run_task_id)),
                do_escape=False
            )
