# -*- coding: utf-8 -*-
from sandbox import sdk2
from sandbox import common

from sandbox.common.types import client as ctc
from sandbox.common.types import task as ctt

from sandbox.projects.hollywood.common import resources
from sandbox.projects.common.build import parameters as build_parameters
from sandbox.projects.common.nanny import nanny


class BuildHollywoodPackageAndRelease(nanny.ReleaseToNannyTask2, sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        cores = 56

    class Parameters(sdk2.Task.Parameters):
        package_resource_type = sdk2.parameters.String(
            'Ya package resource type',
            required=True,
            default='HOLLYWOOD_PACKAGE',
        )

        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl(
            "checkout_arcadia_from_url",
            required=True,
            default_value='arcadia:/arc/trunk/arcadia'
        )

        package_path = sdk2.parameters.String(
            'Arcadia path to package.json',
            required=True,
            default='alice/hollywood/shards/common/prod/hollywood_package.json',
        )

        with sdk2.parameters.String('Release type', required=True) as release_type:
            release_type.values['stable'] = 'stable'
            release_type.values['prestable'] = 'prestable'
            release_type.values['testing'] = 'testing'
            release_type.values['unstable'] = 'unstable'

        copy_trunk = sdk2.parameters.Bool(
            'Try to checkout branch by copying from trunk',
            required=False,
            default=False
        )

        run_tests = sdk2.parameters.Bool(
            'Run arcadia tests',
            required=False,
            default=False
        )

        use_aapi_fuse = build_parameters.UseArcadiaApiFuse(default_value=True)
        aapi_fallback = build_parameters.AllowArcadiaApiFallback(default_value=True)

        with sdk2.parameters.Output:
            hollywood_package = sdk2.parameters.Resource(
                "Hollywood package",
                resource_type=resources.HollywoodPackage,
                required=True,
            )

    class Context(sdk2.Task.Context):
        hollywood_package_id = None
        hollywood_package_resource_id = None

    def on_execute(self):
        if not self.Context.hollywood_package_id:
            subtask = sdk2.Task['BUILD_HOLLYWOOD_PACKAGE'](
                self,
                aapi_fallback=self.Parameters.aapi_fallback,
                build_type='relwithdebinfo',
                checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
                copy_trunk=self.Parameters.copy_trunk,
                description='Build Hollywood package',
                owner=self.owner,
                packages=self.Parameters.package_path,
                raw_package=False,
                resource_type=self.Parameters.package_resource_type,
                run_tests=self.Parameters.run_tests,
                use_aapi_fuse=self.Parameters.use_aapi_fuse,
            ).enqueue()
            self.Context.hollywood_package_id = subtask.id
            raise sdk2.WaitTask([subtask], ctt.Status.Group.FINISH + ctt.Status.Group.BREAK, True)

        subtask = self.find(id=self.Context.hollywood_package_id).first()
        if subtask.status in ctt.Status.Group.SCHEDULER_FAILURE:
            raise common.errors.TaskError('Subtask {} ({}) was finished with the status of {}'.format(subtask.type, subtask.id, subtask.status))

        self.Parameters.hollywood_package = resources.HollywoodPackage.find(task=subtask).first()
        self.Context.hollywood_package_resource_id = self.Parameters.hollywood_package.id

        if (self.Parameters.release_type == "stable"):
            release_type = ctt.ReleaseStatus.SATBLE
        elif (self.Parameters.release_type == "prestable"):
            release_type = ctt.ReleaseStatus.PRESTABLE
        elif (self.Parameters.release_type == "testing"):
            release_type = ctt.ReleaseStatus.TESTING
        elif (self.Parameters.release_type == "unstable"):
            release_type = ctt.ReleaseStatus.UNSTABLE
        else:
            release_type = None

        if (release_type is not None):
            sdk2.Task.server.release(
                task_id=self.Context.hollywood_package_id,
                type=release_type,
                subject=self.Parameters.description
            )
