# -*- coding: utf-8 -*-

import json
import logging
import requests

from sandbox import sdk2
import sandbox.projects.release_machine.core.task_env as task_env

from sandbox.projects.hollywood.common.const import HollywoodConsts
from sandbox.projects.hollywood.common.nanny import HollywoodNannyToolbox
from sandbox.projects.hollywood.common.yappy import HollywoodYappyToolbox

from sandbox.projects.alice_evo.common.const import AccessTokens
from sandbox.projects.alice_evo.common.release_machine import ReleaseMachineToolbox
from sandbox.projects.alice_evo.common.startrek import StartrekToolbox


logger = logging.getLogger(__name__)


def create_beta_name(shard, ctx):
    if shard == HollywoodConsts.COMMON_SHARD:
        return "alice-{}-{}-rc-{}-{}".format(
            ctx.ctx_component_name,
            "common",
            ctx.ctx_branch_num,
            ctx.ctx_tag_num
        )
    elif shard == HollywoodConsts.GENERAL_CONVERSATION_SHARD:
        return "alice-hollywood-general-conversation-branch-{}".format(ctx.ctx_branch_num)
    elif shard == HollywoodConsts.GOODWIN_SHARD:
        return "alice-goodwin-rm-{}-{}".format(
            ctx.ctx_branch_num,
            ctx.ctx_tag_num
        )

    return "alice-{}-{}-rc-{}-{}".format(
        ctx.ctx_component_name,
        ctx.ctx_shard_name,
        ctx.ctx_branch_num,
        ctx.ctx_tag_num
    )


def get_hamster_perf_slot(hamster_slots, beta_slot):
    beta_dc = beta_slot.split(".")[1]
    for slot in hamster_slots:
        if beta_dc == slot.split(".")[1]:
            return slot


def get_firmware_version(platform):
    resp = requests.get("http://int.quasar.yandex.ru/admin/get_group_update?platform={platform}&group=default".format(platform=platform), verify=False).json()
    if "version" in resp:
        return str(resp["version"])
    return ""


def get_system_config(platform):
    resp = requests.get("http://int.quasar.yandex.ru/admin/get_system_config?platform={platform}&group=default".format(platform=platform), verify=False)
    return resp.json()


class GetHollywoodYappyBetaInfo(sdk2.Task):
    """
        Context for Hollywood and Alice Evo tests in release machine
    """

    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.startrek_client

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 600  # 10 min
        in_branch_num = sdk2.parameters.String(
            "in_branch_num",
            required=True
        )
        in_tag_num = sdk2.parameters.String(
            "in_tag_num",
            required=True
        )
        shard = sdk2.parameters.String(
            "shard",
            required=True
        )
        with sdk2.parameters.Output:
            out_branch_num = sdk2.parameters.String(
                "out_branch_num",
                required=True
            )
            out_tag_num = sdk2.parameters.String(
                "out_tag_num",
                required=True
            )
            out_release_id = sdk2.parameters.String(
                "out_release_id"
            )
            out_release_ticket = sdk2.parameters.String(
                "out_release_ticket"
            )
            out_assessors_ticket = sdk2.parameters.String(
                "out_assessors_ticket"
            )
            out_beta_name = sdk2.parameters.String(
                "out_beta_name"
            )
            out_beta_balancer_url = sdk2.parameters.String(
                "out_beta_balencer_url"
            )
            out_beta_slots = sdk2.parameters.List(
                "out_beta_slots"
            )
            out_beta_perf_slot = sdk2.parameters.String(
                "out_beta_perf_slot"
            )
            out_vins_url = sdk2.parameters.String(
                "out_vins_url"
            )
            out_vins_hamster_url = sdk2.parameters.String(
                "out_vins_hamster_url"
            )
            out_experiments = sdk2.parameters.List(
                "out_experiments"
            )
            out_hamster_slots = sdk2.parameters.List(
                "out_hamster_slots"
            )
            out_hamster_no_fast_data_slots = sdk2.parameters.List(
                "out_hamster_no_fast_data_slots"
            )
            out_hamster_no_fast_data_perf_slot = sdk2.parameters.String(
                "out_hamster_no_fast_data_perf_slot"
            )
            out_yandex_station_firmware_version = sdk2.parameters.String(
                "out_yandex_station_firmware_version"
            )
            out_yandex_mini_firmware_version = sdk2.parameters.String(
                "out_yandex_mini_firmware_version"
            )
            out_yandex_station_config = sdk2.parameters.String(
                "out_yandex_station_config"
            )
            out_yandex_mini_config = sdk2.parameters.String(
                "out_yandex_mini_config"
            )

    def on_execute(self):
        logger.info("[INFO] Starting task")

        self.Context.ctx_shard_constants = HollywoodConsts.SHARD_CONSTANTS[self.Parameters.shard]
        self.Context.ctx_component_name = self.Context.ctx_shard_constants["release_machine_name"]
        self.Context.ctx_shard_name = self.Parameters.shard

        # branch_num
        self.Parameters.out_branch_num = self.Parameters.in_branch_num
        self.Context.ctx_branch_num = self.Parameters.out_branch_num

        # tag_num
        self.Parameters.out_tag_num = self.Parameters.in_tag_num
        self.Context.ctx_tag_num = self.Parameters.out_tag_num

        # release_id
        self.Parameters.out_release_id = "{}-{}-{}".format(
            self.Context.ctx_component_name,
            self.Context.ctx_branch_num,
            self.Context.ctx_tag_num
        )
        self.Context.ctx_release_id = self.Parameters.out_release_id

        # release_ticket
        self.Parameters.out_release_ticket = ReleaseMachineToolbox.get_release_ticket_for_branch(
            self.Context.ctx_component_name,
            self.Context.ctx_branch_num
        )
        self.Context.ctx_release_ticket = self.Parameters.out_release_ticket

        # linked assessors ticket
        from startrek_client import Startrek
        st_client = Startrek(token=StartrekToolbox.get_startrek_token(),
                             useragent=AccessTokens.ST_USERAGENT)
        self.Parameters.out_assessors_ticket = StartrekToolbox.get_linked_assessors_ticket(
            st_client,
            self.Context.ctx_release_ticket
        )
        self.Context.ctx_assessors_ticket = self.Parameters.out_assessors_ticket

        # beta_name
        self.Parameters.out_beta_name = create_beta_name(self.Parameters.shard, self.Context)
        self.Context.ctx_beta_name = self.Parameters.out_beta_name

        # beta_balancer_url
        self.Parameters.out_beta_balancer_url = "http://{}.yappy.beta.yandex.ru".format(
            self.Context.ctx_beta_name
        )
        self.Context.ctx_beta_balancer_url = self.Parameters.out_beta_balancer_url

        # beta_slots
        yappy_token = sdk2.Vault.data(HollywoodConsts.YAPPY_TOKEN_OWNER, HollywoodConsts.YAPPY_TOKEN_NAME)
        self.Parameters.out_beta_slots = HollywoodYappyToolbox.get_yappy_slots(
            self.Context.ctx_beta_name,
            yappy_token,
            logger,
            self.Context.ctx_shard_constants["beta_type"]
        )
        self.Context.ctx_beta_slots = self.Parameters.out_beta_slots

        # beta_perf_slot
        self.Parameters.out_beta_perf_slot = self.Context.ctx_beta_slots[0][:-1] + "1"  # adjust port (80 -> 81)
        self.Context.ctx_beta_perf_slot = self.Parameters.out_beta_perf_slot

        # vins_url
        self.Parameters.out_vins_url = self.Context.ctx_beta_balancer_url + "/speechkit/app/pa/"
        self.Context.ctx_vins_url = self.Parameters.out_vins_url

        # vins_hamster_url
        self.Parameters.out_vins_hamster_url = self.Context.ctx_beta_balancer_url + "/speechkit/app/pa/"
        self.Context.ctx_vins_hamster_url = self.Parameters.out_vins_hamster_url

        # experiments
        self.Parameters.out_experiments = list(set().union(*self.Context.ctx_shard_constants["scenarios_experiments"].values()))
        self.Context.ctx_experiments = self.Parameters.out_experiments

        # hamster_slots
        self.Parameters.out_hamster_slots = [inst[0] for inst in HollywoodNannyToolbox.get_service_instances("hollywood_hamster")]
        self.Context.ctx_hamster_slots = self.Parameters.out_hamster_slots

        # hamster_no_fast_data_slots
        self.Parameters.out_hamster_no_fast_data_slots = [inst[0] for inst in HollywoodNannyToolbox.get_service_instances("hollywood_hamster_no_fast_data")]
        self.Context.ctx_hamster_no_fast_data_slots = self.Parameters.out_hamster_no_fast_data_slots

        # hamster_no_fast_data_perf_slot
        hamster_no_fast_data_perf_slot = get_hamster_perf_slot(self.Context.ctx_hamster_no_fast_data_slots, self.Context.ctx_beta_perf_slot)
        self.Parameters.out_hamster_no_fast_data_perf_slot = hamster_no_fast_data_perf_slot + ":81"
        self.Context.ctx_hamster_no_fast_data_perf_slot = self.Parameters.out_hamster_no_fast_data_perf_slot

        # firmware_version_yandex_station
        self.Parameters.out_yandex_station_firmware_version = get_firmware_version("yandexstation")
        self.Context.ctx_yandex_station_firmware_version = self.Parameters.out_yandex_station_firmware_version

        # firmware_version_yandex_mini
        self.Parameters.out_yandex_mini_firmware_version = get_firmware_version("yandexmini")
        self.Context.ctx_yandex_mini_firmware_version = self.Parameters.out_yandex_mini_firmware_version

        # out_yandex_station_config
        yandex_station_config = get_system_config("yandexstation")
        yandex_station_config["vinsUrl"] = self.Context.ctx_vins_url
        self.Parameters.out_yandex_station_config = json.dumps(yandex_station_config)
        self.Context.ctx_yandex_station_config = self.Parameters.out_yandex_station_config

        # out_yandex_mini_config
        yandex_mini_config = get_system_config("yandexmini")
        yandex_mini_config["vinsUrl"] = self.Context.ctx_vins_url
        self.Parameters.out_yandex_mini_config = json.dumps(yandex_mini_config)
        self.Context.ctx_yandex_mini_config = self.Parameters.out_yandex_mini_config

        logger.info("[INFO] Finishing task")
