from os.path import isfile, join
import json
import os

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.paths import get_logs_folder
import sandbox.common.types.client as ctc

from sandbox.projects.vins.common.resources import MegamindShooterBinary
from sandbox.projects.hollywood.common.resources import HollywoodPackage, HollywoodAmmo, HollywoodDiff


class HollywoodDiffTest(sdk2.Task):
    ''' Do a diff test on two Hollywood revisions '''

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        cores = 20

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Packages') as packages_parameters:
            old_hollywood_package = sdk2.parameters.Resource(
                'Old Hollywood package',
                resource_type=HollywoodPackage,
                required=True
            )

            new_hollywood_package = sdk2.parameters.Resource(
                'New Hollywood package',
                resource_type=HollywoodPackage,
                required=True
            )

            hollywood_ammo = sdk2.parameters.Resource(
                'Hollywood ammo',
                resource_type=HollywoodAmmo,
                required=True
            )

            shooter_binary = sdk2.parameters.Resource(
                'Shooter binary resource',
                resource_type=MegamindShooterBinary,
                required=True
            )

        with sdk2.parameters.Group('Config') as config_parameters:
            requests_limit = sdk2.parameters.Integer(
                'Requests limit',
                default=1000000,
                required=True
            )

            threads = sdk2.parameters.Integer(
                'Shooting threads',
                default=15,
                required=True
            )

        kill_timeout = 2 * 60 * 60  # 2 hours

    @staticmethod
    def unpack_hollywood_package(hollywood_package, folder_name):
        hollywood_package_dir = join(os.getcwd(), folder_name)
        os.mkdir(hollywood_package_dir)

        hollywood_archive_path = str(sdk2.ResourceData(hollywood_package).path)
        sp.Popen(['tar', 'xzf', hollywood_archive_path, '-C', hollywood_package_dir]).wait()

        return hollywood_package_dir

    # unpack ammo to separated files: one ammo = one file
    # shooter binary accepts only one ammo per file
    @staticmethod
    def unpack_ammo(ammo_package):
        ammo_dir = str(sdk2.ResourceData(ammo_package).path)

        unpacked_ammo_dir = join(os.getcwd(), 'ammo_dir')
        os.mkdir(unpacked_ammo_dir)

        ammo_files_path = [join(ammo_dir, f) for f in os.listdir(ammo_dir) if isfile(join(ammo_dir, f))]

        ammo_number = 0
        for ammo_file_path in ammo_files_path:
            with open(ammo_file_path, 'r') as ammo_file:
                for line in ammo_file:
                    unpacked_ammo_file_path = join(unpacked_ammo_dir, '{:05d}.json'.format(ammo_number))
                    with open(unpacked_ammo_file_path, 'w') as unpacked_ammo_file:
                        unpacked_ammo_file.write(line)
                    ammo_number += 1

        return unpacked_ammo_dir

    def on_execute(self):
        # load resources
        old_hollywood_package_dir = self.unpack_hollywood_package(self.Parameters.old_hollywood_package, 'old_package')
        new_hollywood_package_dir = self.unpack_hollywood_package(self.Parameters.new_hollywood_package, 'new_package')
        ammo_dir = self.unpack_ammo(self.Parameters.hollywood_ammo)
        shooter_binary = str(sdk2.ResourceData(self.Parameters.shooter_binary).path)

        # setup config for shooter
        old_hollywood_responses = join(get_logs_folder(), 'old_responses')
        new_hollywood_responses = join(get_logs_folder(), 'new_responses')

        config_path = join(get_logs_folder(), 'config.json')
        config = {
            'runs_settings': [
                {
                    'package_path': old_hollywood_package_dir,
                    'logs_path': join(get_logs_folder(), 'old_shoot_logs'),
                    'responses_path': old_hollywood_responses,
                    'enable_hollywood_mode': True
                },
                {
                    'package_path': new_hollywood_package_dir,
                    'logs_path': join(get_logs_folder(), 'new_shoot_logs'),
                    'responses_path': new_hollywood_responses,
                    'enable_hollywood_mode': True
                }
            ],

            'requests_path': ammo_dir,
            'requests_limit': self.Parameters.requests_limit,
            'threads': self.Parameters.threads
        }
        with open(config_path, 'w') as f:
            json.dump(config, f)

        # launch shooter runner
        out_path = join(get_logs_folder(), 'shooter_run.out')
        err_path = join(get_logs_folder(), 'shooter_run.err')

        with open(out_path, 'w') as out, open(err_path, 'w') as err:
            sp.Popen(
                [
                    shooter_binary, 'run', config_path
                ],
                stdout=out, stderr=err, stdin=None
            ).wait()

        # prepare resource
        resource = sdk2.ResourceData(HollywoodDiff(
            task=self,
            description='Responses from HollywoodDiffTest task #{}'.format(self.id),
            path=join(os.getcwd(), 'results')
        ))

        # launch shooter differ
        out_path = join(get_logs_folder(), 'shooter_diff.out')
        err_path = join(get_logs_folder(), 'shooter_diff.err')

        diff_stats_path = join(str(resource.path), 'diff_stats.json')  # stats about diff
        with open(out_path, 'w') as out, open(err_path, 'w') as err:
            sp.Popen(
                [
                    shooter_binary, 'diff',
                    '--mode', 'hollywood',
                    '--diffs-per-file', '20',
                    '--old-path', old_hollywood_responses,
                    '--new-path', new_hollywood_responses,
                    '--output-path', str(resource.path),
                    '--stats-path', diff_stats_path
                ],
                stdout=out, stderr=err, stdin=None
            ).wait()

        resource.ready()

        # requirement for diff alerting
        with open(diff_stats_path, 'r') as f:
            diff_stats = json.load(f)
        diff_count = diff_stats['responses_diffs_count']

        self.Context.has_diff = diff_count > 0

    @sdk2.header()
    def header(self):
        if self.Context.has_diff:
            return 'There are diffs! Watch diff resource'
        return 'No diffs found'
