import os
import yaml

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.hollywood.common import resources
from sandbox.projects.hollywood.common.const import HollywoodConsts


def prepare_experiments(experiments):
    if not isinstance(experiments, list):
        return []
    res = []
    for exp in experiments:
        res.append(str(exp + "=1"))
    return res


def prepare_config(config_file_path, vins_url, experiments):
    config = {
        "yt": {
            "token": "robot-alice-priemka_yt_token",
            "mr_account": "alice"
        },
        "yql": {
            "token": "robot-alice-priemka_yql_token"
        },
        "sandbox": {
            "token": "robot-alice-priemka_sandbox_token"
        },
        "soy": {
            "token": "robot-alicesoyaccept-soy-token"
        },
        "alice_common": {
            "ya_plus_robot_token": "robot-alice-vins_oauth_token",
            "scraper_over_yt_pool": "alice-acceptance"
        },
        "screenshots_eval": {
            "vins_url_2": "http://megamind.alice.yandex.net/rc/speechkit/app/pa/",
            "bass_url_2": "http://bass.hamster.alice.yandex.net/"
        },
        "acceptance_order": [
            ["check_analytics_info"]
        ],
        "check_analytics_info": {
            "input_table": "//home/alice/toloka/accept/ue2e_quasar/basket",
            "test_vins_url": str(vins_url),
            "experiments": prepare_experiments(experiments) + ["analytics_info=1", "tunneller_analytics_info=1"],
            "revision": 6228393
        }
    }
    with open(config_file_path, "w") as f:
        yaml.dump(config, f)


class RunAnalyticsInfoTests(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 10 * 60  # 10 minutes is more than enough

        vins_url = sdk2.parameters.String(
            "Vins url",
            required=True,
        )

        experiments = sdk2.parameters.List(
            "list of experiments",
            default=[]
        )

        analytics_info_tests_binary = sdk2.parameters.Resource(
            "AnalyticsInfo tests binary",
            resource_type=resources.AnalyticsInfoTestsBinary,
            required=True
        )

        no_exec = sdk2.parameters.Bool(
            "Do not start the graph",
            default=False
        )

        with sdk2.parameters.Output:
            nirvana_graph_url = sdk2.parameters.String("Nirvana graph url")
            config_file = sdk2.parameters.Resource("AnalyticsInfo tests config", resource_type=resources.AnalyticsInfoTestsConfig)
            tests_logs = sdk2.parameters.Resource("AnalyticsInfo tests logs", resource_type=resources.AnalyticsInfoTestsLogs)

    def on_execute(self):
        self.Parameters.config_file = resources.AnalyticsInfoTestsConfig(
            self, "AnalyticsInfo tests config file for " + self.Parameters.description, "config.yaml", ttl=90
        )
        config_file_path = str(sdk2.ResourceData(self.Parameters.config_file).path)

        with self.memoize_stage.prepare_config:
            vins_url = self.Parameters.vins_url
            experiments = self.Parameters.experiments
            prepare_config(config_file_path, vins_url, experiments)

        with self.memoize_stage.run_analytics_info_tests:
            nirvana_oauth_token = sdk2.Vault.data(
                HollywoodConsts.NIRVANA_PRIEMKINS_TOKEN_OWNER,
                HollywoodConsts.NIRVANA_PRIEMKINS_TOKEN_NAME
            )
            os.environ["NIRVANA_TOKEN"] = nirvana_oauth_token
            cmd = [
                str(sdk2.ResourceData(self.Parameters.analytics_info_tests_binary).path) + "/run",
                "--user-config", config_file_path,
            ]
            if self.Parameters.no_exec:
                cmd.append("--no-exec")

            self.Parameters.tests_logs = resources.AnalyticsInfoTestsLogs(
                self, "AnalyticsInfo tests logs" + self.Parameters.description, "analytics_info_tests_logs.txt", ttl=90
            )
            logs_file_path = str(sdk2.ResourceData(self.Parameters.tests_logs).path)
            logs_file = open(logs_file_path, "w")
            sp.Popen(" ".join(cmd), shell=True, stdout=logs_file, stderr=logs_file).wait()
            logs_file.close()

            logs = open(logs_file_path, "r").read()
            for line in logs.split():
                if "https://nirvana.yandex-team.ru" in line:
                    self.Parameters.nirvana_graph_url = line

            self.set_info("full AnalyticsInfo tests stdout:\n" + logs)
        self.set_info("pid {}: finished {}".format(os.getpid(), cmd))
