import os
import yaml

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.hollywood.common import resources
from sandbox.projects.hollywood.common.const import HollywoodConsts


def prepare_experiments(experiments):
    if not isinstance(experiments, list):
        return []
    res = []
    for exp in experiments:
        res.append(str(exp + "=1"))
    return res


def prepare_config(config_file_path, vins_url, experiments):
    config = {
        "nirvana": {
            "quota": "voice-core",
            "hitman_guid": "41f39437-f9fc-4ecf-b0de-7c8729a506b3"
        },
        "yt": {
            "token": "robot-alice-priemka_yt_token",
            "proxy": "hahn",
            "pool": "",
            "mr_account": "alice",
            "mr_output_ttl": 1,
            "mr_output_path": "home[#if param[\"mr-account\"] != meta.owner]/${param[\"mr-account\"]}[/#if]/${meta.owner}/nirvana/marker_tests/${meta.operation_uid}"
        },
        "yql": {
            "token": "robot-alice-priemka_yql_token"
        },
        "arcanum": {
            "token": "robot-alice-priemka_arcanum_token"
        },
        "soy": {
            "token": "robot-alicesoyaccept-soy-token"
        },
        "alice": {
            "uniproxy_url": "ws://uniproxy-kpi.alice.yandex.net/uni.ws",
            "vins_url": str(vins_url),
            "oauth": "bass_testing_analyst_token",
            "scraper_timeout": "12h",
            "scraper_pool": "alice-acceptance",
            "asr_chunk_size": -1,
            "experiments": prepare_experiments(experiments)
        }
    }
    with open(config_file_path, "w") as f:
        yaml.dump(config, f)


class RunMarkerTests(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 10 * 60  # 10 minutes is more than enough

        vins_url = sdk2.parameters.String(
            "Vins url",
            required=True,
        )

        experiments = sdk2.parameters.List(
            "list of experiments",
            default=[]
        )

        marker_tests_binary = sdk2.parameters.Resource(
            "Marker tests binary",
            resource_type=resources.MarkerTestsBinary,
            required=True
        )

        no_exec = sdk2.parameters.Bool(
            "Do not start the graph",
            default=False
        )

        use_soy_testing = sdk2.parameters.Bool(
            "Use SoY testing contour. SCRAPEROVERYT-234",
            default=False
        )

        wait_nirvana = sdk2.parameters.Bool(
            "Wait nirvana graph to execute",
            default=False
        )

        with sdk2.parameters.Output:
            nirvana_graph_url = sdk2.parameters.String("Nirvana graph url")
            config_file = sdk2.parameters.Resource("Marker tests config", resource_type=resources.MarkerTestsConfig)
            marker_tests_logs = sdk2.parameters.Resource("Marker tests logs", resource_type=resources.MarkerTestsLogs)

    def on_execute(self):
        self.Parameters.config_file = resources.MarkerTestsConfig(
            self, "Marker tests config file for " + self.Parameters.description, "config.yaml", ttl=90
        )
        config_file_path = str(sdk2.ResourceData(self.Parameters.config_file).path)

        with self.memoize_stage.prepare_config:
            vins_url = self.Parameters.vins_url
            experiments = self.Parameters.experiments
            prepare_config(config_file_path, vins_url, experiments)

        with self.memoize_stage.run_marker_tests:
            nirvana_oauth_token = sdk2.Vault.data(
                HollywoodConsts.NIRVANA_PRIEMKINS_TOKEN_OWNER,
                HollywoodConsts.NIRVANA_PRIEMKINS_TOKEN_NAME
            )
            os.environ["NIRVANA_TOKEN"] = nirvana_oauth_token
            cmd = [
                str(sdk2.ResourceData(self.Parameters.marker_tests_binary).path) + "/marker_tests",
                "--use-linked-resources",
                "--config-path", config_file_path,
                "--scraper-mode", "soy_test" if self.Parameters.use_soy_testing else "prod"
            ]
            if self.Parameters.wait_nirvana:
                cmd.append("--wait")

            if self.Parameters.no_exec:
                cmd.append("--no-exec")

            self.Parameters.marker_tests_logs = resources.MarkerTestsLogs(
                self, "Marker tests logs" + self.Parameters.description, "marker_tests_logs.txt", ttl=90
            )
            logs_file_path = str(sdk2.ResourceData(self.Parameters.marker_tests_logs).path)
            logs_file = open(logs_file_path, "w")
            sp.Popen(" ".join(cmd), shell=True, stdout=logs_file, stderr=logs_file).wait()
            logs_file.close()

            logs = open(logs_file_path, "r").read()
            url = ''
            for line in logs.split():
                if "https://nirvana.yandex-team.ru" in line:
                    self.Parameters.nirvana_graph_url = line
                    url = line
                if "ERROR" in line:
                    raise Exception("Marker tests running failed, nirvana url: %s" % url)

            self.set_info("full marker tests stdout:\n" + logs)
        self.set_info("pid {}: finished {}".format(os.getpid(), cmd))
