import logging
from sandbox import sdk2

import sandbox.common.errors as errors
from sandbox.common.types.resource import State
from sandbox.common.types.task import Status
from sandbox.projects.common import binary_task
from sandbox.projects.hollywood.common.resources import HollywoodAmmo, AppHostPandoraGunExecutable
from sandbox.projects.tank.load_resources.resources import YANDEX_TANKAPI_LXC_CONTAINER
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi

logger = logging.getLogger(__name__)


def link_format(href, text, description=''):
    return "{}<a href=\"{}\">{}</a>".format(description, href, text)


class StartHollywoodPerfTesting(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Task that starts the perfomance testing of Hollywood, which is a shooting to two targets (stable and testing) with
    a Yandex.Tank and comparing their results with lunapark compare.
    """
    class Requirements(sdk2.Task.Requirements):
        ram = 10 * 1024  # 10 Gb
        disk_space = 30 * 1024  # 10 Gb

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 5 * 60 * 60
        startrek_ticket_id = sdk2.parameters.String(
            "StarTrek ticket id.",
            required=True,
        )
        ammo_resource = sdk2.parameters.Resource(
            "Ammo resource.",
            resource_type=HollywoodAmmo,
            required=True,
        )
        gun_resource = sdk2.parameters.Resource(
            "Pandora gun executable resource. Leave it empty to use the last released:stable.",
            resource_type=AppHostPandoraGunExecutable,
            required=False,
        )
        startup_load_profile = sdk2.parameters.String(
            "Tank startup load profile",
            required=True,
            default="[{type: once, times: 1000}]",
        )
        rps_load_profile = sdk2.parameters.String(
            "Tank rps load profile",
            required=True,
            default="[{type: const, duration: 5m, ops: 10}]",
        )
        testing_target_url = sdk2.parameters.String(
            "Testing target host:port.",
            required=True,
        )
        stable_target_url = sdk2.parameters.String(
            "Stable target host:port.",
            description="Leave it empty if you do not want to compare the shootings.",
            required=False,
        )
        ext_params = binary_task.LastBinaryReleaseParameters(stable=True)

        with sdk2.parameters.Output:
            testing_lunapark_job_id = sdk2.parameters.String('Testing lunapark job id.', default_value='')
            testing_lunapark_url = sdk2.parameters.String('Testing lunapark url.', default_value='')
            stable_lunapark_job_id = sdk2.parameters.String('Stable lunapark job id.', default_value='')
            stable_lunapark_url = sdk2.parameters.String('Stable lunapark url.', default_value='')
            compare_url = sdk2.parameters.String('Testing vs Stable shootings compare url.', default_value='')

    def on_save(self):
        super(StartHollywoodPerfTesting, self).on_save()

    def on_execute(self):
        super(StartHollywoodPerfTesting, self).on_execute()

        with self.memoize_stage.find_tank_lxc_container:
            self._container = YANDEX_TANKAPI_LXC_CONTAINER.find(state=State.READY).order(-YANDEX_TANKAPI_LXC_CONTAINER.id).first().id
            logger.info("Found a YANDEX_TANKAPI_LXC_CONTAINER, id={}".format(str(self._container)))

        with self.memoize_stage.find_all_tanks:
            from alice.hollywood.library.python.perf_testing.nanny_finder import find_containers_in_nanny_group
            self.Context._all_tanks = find_containers_in_nanny_group("hollywood-tank")

        with self.memoize_stage.start_shootings:
            self.Context._testing_shooting_id = self._start_shooting(self.Parameters.testing_target_url, "TESTING")

            self.Context._stable_shooting_id = None
            if self.Parameters.stable_target_url != "":
                self.Context._stable_shooting_id = self._start_shooting(self.Parameters.stable_target_url, "STABLE")

        with self.memoize_stage.wait_shootings:
            ids_to_wait = filter(lambda id: id is not None, [self.Context._testing_shooting_id, self.Context._stable_shooting_id])
            raise sdk2.WaitTask(ids_to_wait, Status.Group.FINISH | Status.Group.BREAK, wait_all=True, timeout=14400)

        with self.memoize_stage.find_shooting_results:
            testing_result = self._find_shooting_result(self.Context._testing_shooting_id)
            self.Parameters.testing_lunapark_job_id = testing_result.Parameters.lunapark_job_id
            self.Parameters.testing_lunapark_url = testing_result.Parameters.lunapark_link
            self.set_info(
                link_format(self.Parameters.testing_lunapark_url, self.Parameters.testing_lunapark_url, description='Testing shooting: '),
                do_escape=False
            )

            if self.Context._stable_shooting_id is not None:
                stable_result = self._find_shooting_result(self.Context._stable_shooting_id)
                self.Parameters.stable_lunapark_job_id = stable_result.Parameters.lunapark_job_id
                self.Parameters.stable_lunapark_url = stable_result.Parameters.lunapark_link
                self.set_info(
                    link_format(self.Parameters.stable_lunapark_url, self.Parameters.stable_lunapark_url, description='Stable shooting: '),
                    do_escape=False
                )

                self.Parameters.compare_url = "https://lunapark.yandex-team.ru/compare/" \
                                              "#jobs={stable_id},{testing_id}&tab=test_data&mainjob={stable_id}" \
                                              "&helper=all&cases=&plotGroup=additional&metricGroup=&target=" \
                                              .format(testing_id=self.Parameters.testing_lunapark_job_id,
                                                      stable_id=self.Parameters.stable_lunapark_job_id)
                self.set_info(
                    link_format(self.Parameters.compare_url, "url", description='Stable vs Testing compare: '),
                    do_escape=False
                )

    def _start_shooting(self, target_url, tag):
        tanks = self._find_tanks_in_same_dc(target_url)
        logger.info("Tanks to be used: {}".format(str(tanks)))

        gun_resource = self.Parameters.gun_resource
        if gun_resource is None:
            gun_resource = AppHostPandoraGunExecutable.find(
                state=State.READY,
                attrs=dict(
                    released='stable',
                ),
            ).first()

        config_content = self._make_config_content(target_url, tag, gun_resource)

        logger.info("Preparing shooting task...")
        shooting_task = ShootViaTankapi(
            self,
            description="Shooting to {} target {}".format(tag, target_url),
            use_public_tanks=False,
            tanks=tanks,
            ammo_source="in_config",
            config_source="file",
            config_content=config_content,
            container=self._container,
        ).enqueue()
        logger.info("Started shooting task id={}".format(str(shooting_task.id)))
        return shooting_task.id

    def _find_shooting_result(self, shooting_task_id):
        logger.info("Searching for shooting result by id={}".format(shooting_task_id))
        shooting_result = sdk2.Task.find(
            id=shooting_task_id,
            children=True
        ).first()
        logger.info("Shooting result found, id={}".format(str(shooting_result.id)))
        if shooting_result.status not in Status.Group.SUCCEED:
            raise errors.TaskError("Child task did not succeed.")
        return shooting_result

    def _find_tanks_in_same_dc(self, target_url):
        dc = self._detect_dc(target_url)
        tank_infos = self.Context._all_tanks[dc]
        return ["{host}:{port}".format(host=tank_info["host"], port=tank_info["port"]) for tank_info in tank_infos]

    def _detect_dc(self, target_url):
        if ".man.yp-c.yandex.net" in target_url:
            return "man"
        if ".sas.yp-c.yandex.net" in target_url:
            return "sas"
        if ".vla.yp-c.yandex.net" in target_url:
            return "vla"
        raise errors.TaskError("Could not detect DC of target url={}".format(target_url))

    def _make_config_content(self, target_url, tag, gun_resource):
        config_template = """
phantom:
    enabled: false
    address: {target_url}
uploader:
    operator: lunapark
    task: {startrek_ticket_id}
    job_name: {tag}
    job_dsc: 'started from Sandbox'
    package: yandextank.plugins.DataUploader
neuploader:
    enabled: false
pandora:
    enabled: true
    expvar: false
    pandora_cmd: {gun_download_url}
    resources:
      - src: {ammo_download_url}
        dst: ./ammo.jsonlines
    config_content:
        pools:
          - id: example
            gun:
                type: app_host_pandora_gun
                target: {target_url}
                app_host_servant_method: Invoke
            ammo:
                type: app_host_provider
                source:
                    type: file
                    path: ./ammo.jsonlines
            result:
                type: phout
                destination: ./phout.log
            startup:
              {startup_load_profile}
            rps:
              {rps_load_profile}
        log:
            level: error
        monitoring:
            expvar:
                enabled: true
                port: 1234
"""
        return config_template.format(
            target_url=target_url,
            ammo_download_url="{}/ammo.out".format(self.Parameters.ammo_resource.http_proxy),
            gun_download_url=gun_resource.http_proxy,
            startup_load_profile=self.Parameters.startup_load_profile,
            rps_load_profile=self.Parameters.rps_load_profile,
            startrek_ticket_id=self.Parameters.startrek_ticket_id,
            tag=tag,
        )
