# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2
import sandbox.projects.release_machine.core.task_env as task_env

from sandbox.projects.alice_evo.common.const import AccessTokens
from sandbox.projects.alice_evo.common.startrek import StartrekToolbox


logger = logging.getLogger(__name__)


def _generate_config_info(info):
    firmware_comment = "Firmware version: %%{}%%".format(info["version"])
    config_comment = "Config: %%{}%%".format(info["config"])
    return firmware_comment + "\n" + config_comment


def _generate_alice_assessors_comment(release_id, vins_url, station_info, mini_info):
    url_comment = "  VINS URL: %%{}%%".format(vins_url.replace("http://vins.hamster.alice.yandex.net", "http://vins.alice.yandex.net", 1))
    station_comment = "  <{{Station Config:\n{}}}>".format(_generate_config_info(station_info))
    mini_comment = "  <{{Station Mini Config:\n{}}}>".format(_generate_config_info(mini_info))

    comment = "<{{{}\n{}\n{}\n{}}}>".format(release_id, url_comment, station_comment, mini_comment)
    return comment


class UpdateHollywoodReleaseTickets(sdk2.Task):
    """
        Task for updating ALICERELEASE and ALICEASSESSORS tickets
    """

    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.startrek_client

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 600  # 10 min
        in_release_id = sdk2.parameters.String(
            "in_release_id",
            required=True
        )
        in_release_ticket = sdk2.parameters.String(
            "in_release_ticket"
        )
        in_assessors_ticket = sdk2.parameters.String(
            "in_assessors_ticket"
        )
        in_vins_url = sdk2.parameters.String(
            "in_vins_url"
        )
        in_yandex_station_firmware_version = sdk2.parameters.String(
            "in_yandex_station_firmware_version"
        )
        in_yandex_mini_firmware_version = sdk2.parameters.String(
            "in_yandex_mini_firmware_version"
        )
        in_yandex_station_config = sdk2.parameters.String(
            "in_yandex_station_config"
        )
        in_yandex_mini_config = sdk2.parameters.String(
            "in_yandex_mini_config"
        )

    def on_execute(self):
        logger.info("Starting task")

        station_info = {
            "version": self.Parameters.in_yandex_station_firmware_version,
            "config": self.Parameters.in_yandex_station_config
        }
        mini_info = {
            "version": self.Parameters.in_yandex_mini_firmware_version,
            "config": self.Parameters.in_yandex_mini_config
        }

        from startrek_client import Startrek
        st_client = Startrek(token=StartrekToolbox.get_startrek_token(),
                                useragent=AccessTokens.ST_USERAGENT)

        vins_url_comment = _generate_alice_assessors_comment(
            self.Parameters.in_release_id,
            self.Parameters.in_vins_url,
            station_info,
            mini_info
        )
        StartrekToolbox.append_startrek_comment(
            st_client,
            vins_url_comment,
            self.Parameters.in_assessors_ticket,
            "VinsUrl for manual testing",
            False
        )
        logger.info("Posted vins_url to ALICEASSESSORS ticket")

        StartrekToolbox.append_startrek_comment(
            st_client,
            vins_url_comment,
            self.Parameters.in_release_ticket,
            "VinsUrl for manual testing",
            False
        )
        logger.info("Posted vins_url to ALICERELEASE ticket")

        logger.info('Finish get_run_result')
