# -*- coding: utf-8 -*-

from config import deploy_settings, default_deploy_configs, default_hamster_deploy_configs

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sdk2 import svn

from sandbox.common import errors
from sandbox.common.types.task import ReleaseStatus
from sandbox.common.types.task import Status

from sandbox.projects.hollywood.common import resources as resources
from sandbox.projects.hollywood.common.const import HollywoodConsts
from sandbox.projects.hollywood.fast_data.BuildHollywoodFastData import BuildHollywoodFastData
from sandbox.projects.hollywood.fast_data.DeployHollywoodFastData import DeployHollywoodFastData

from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import utils
from sandbox.projects.common.build import parameters as build_parameters
from sandbox.projects.websearch.upper.fast_data.ExecutionTimeTracker import ExecutionTimeTracker


def get_arcadia_build_path(shard):
    if shard == HollywoodConsts.COMMON_SHARD:
        return "alice/hollywood/shards/common/prod/fast_data"
    elif shard == HollywoodConsts.GENERAL_CONVERSATION_SHARD:
        return "alice/hollywood/shards/general_conversation/prod/fast_data"
    else:
        return "alice/hollywood/shards/common/prod/fast_data"


def get_arcadia_watch_path(shard):
    if shard == HollywoodConsts.COMMON_SHARD:
        return ["alice/hollywood/shards/common/prod/fast_data"]
    elif shard == HollywoodConsts.GENERAL_CONVERSATION_SHARD:
        return ["alice/begemot/lib/fixlist_index/data/ru",
                "alice/hollywood/library/scenarios/general_conversation/fast_data",
                "alice/hollywood/shards/general_conversation/prod/fast_data"]
    else:
        return ["alice/hollywood/shards/common/prod/fast_data"]


class BuildAndDeployHollywoodFastData(ExecutionTimeTracker):

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 50  # Mb
        environments = [
            environments.PipEnvironment('python-statface-client', version="0.154.0", custom_parameters=["requests==2.18.4"]),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass  # do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 10 * 60  # this task doesn't do anything itself, so 10 minutes is enough

        build_system = build_parameters.BuildSystem()

        with sdk2.parameters.RadioGroup("Deployer mode") as deployer_mode:
            deployer_mode.values['standalone'] = deployer_mode.Value('Run deployer in task', default=True)
            deployer_mode.values['nanny_service'] = deployer_mode.Value('Use nanny service deployer')

            with deployer_mode.value['nanny_service']:
                nanny_service = sdk2.parameters.String('Deployer Nanny service name', required=True)

        use_default_deploy_configs = sdk2.parameters.Bool("Use default deploy config", default=True)
        with use_default_deploy_configs.value[False]:
            deploy_config = sdk2.parameters.JSON("Production deploy config", required=True)
            hamster_deploy_config = sdk2.parameters.JSON("Hamster deploy config", required=True)

        with sdk2.parameters.RadioGroup('Shard') as shard:
            shard.values[HollywoodConsts.COMMON_SHARD] = shard.Value(HollywoodConsts.COMMON_SHARD, default=True)
            shard.values[HollywoodConsts.GENERAL_CONVERSATION_SHARD] = shard.Value(HollywoodConsts.GENERAL_CONVERSATION_SHARD)

        yt_store_token_owner = sdk2.parameters.String("YT_STORE_TOKEN owner", default="VINS")

        release_same_version = sdk2.parameters.Bool("release, if there were no changes in fast data, only peerdirs", default=False)

        only_build = sdk2.parameters.Bool("Only build fast data, do not deploy", default=True)

        skip_tests = sdk2.parameters.Bool("Skip tests, use this at you on risk", default=False)
        with skip_tests.value[True]:
            i_accept = sdk2.parameters.Bool("I understand what can happen", default=False)

        with sdk2.parameters.Output:
            fast_data = sdk2.parameters.Resource(
                "New Hollywood Fast Data",
                resource_type=resources.HollywoodFastData,
                required=True,
            )
            fast_data_bundle = sdk2.parameters.Resource(
                "New Hollywood Fast Data bundle",
                resource_type=resources.HollywoodFastDataBundle,
                required=True,
            )
            last_released_data = sdk2.parameters.Resource(
                "Fallback Hollywood Fast Data bundle",
                resource_type=resources.HollywoodFastDataBundle,
            )
            successfully_deployed_on_hamster = sdk2.parameters.Bool("Successfully deployed on hamster", required=True)
            successfully_rolled_back_hamster = sdk2.parameters.Bool("Successfully rolled back hamster")

            successfully_deployed = sdk2.parameters.Bool("Successfully deployed", required=True)
            successfully_rolled_back = sdk2.parameters.Bool("Successfully rolled back")

    class Context(ExecutionTimeTracker.Context):
        service_group_name_by_task_id = dict()
        deployment_tasks = list()

    @property
    def stage_name(self):
        return 'release'

    def on_execute(self):
        if self.Parameters.use_default_deploy_configs:
            shard = self.Parameters.shard
            self.Context.deploy_config = default_deploy_configs[shard]
            self.Context.hamster_deploy_config = default_hamster_deploy_configs[shard] if shard in default_hamster_deploy_configs else None
        else:
            self.Context.deploy_config = self.Parameters.deploy_config
            self.Context.hamster_deploy_config = self.Parameters.hamster_deploy_config

        with self.memoize_stage.build:
            arcadia_build_path = get_arcadia_build_path(self.Parameters.shard)
            watch_paths = get_arcadia_watch_path(self.Parameters.shard)

            self.Context.revision = max(
                map(
                    int,
                    map(
                        utils.svn_last_change,
                        map(
                            svn.Arcadia.trunk_url,
                            watch_paths
                        )
                    )
                )
            )

            self.Parameters.last_released_data = resources.HollywoodFastDataBundle.find(
                attrs={'released': ReleaseStatus.STABLE, 'shard': self.Parameters.shard}
            ).order(-sdk2.Resource.id).first()

            if self.Parameters.last_released_data is not None:
                self.Context.previous_data_revision = int(self.Parameters.last_released_data.revision)
                self.set_info('Last released data revision: {}'.format(self.Context.previous_data_revision))
            else:
                self.Context.previous_data_revision = 0
            self.set_info('Current revision: {}'.format(self.Context.revision))
            for path in watch_paths:
                self.set_info('Change diff: https://a.yandex-team.ru/arc/diff/trunk/arcadia/{}?prevRev={}&rev={}'.format(
                    path, self.Context.previous_data_revision, self.Context.revision
                ))

            if self.Parameters.last_released_data is not None and self.Context.revision <= self.Parameters.last_released_data.revision:
                self.set_info(
                    'This or newer version of fast data has already been released: {}'.format(lb.resource_link(self.Parameters.last_released_data.id)),
                    do_escape=False
                )
                if not self.Parameters.release_same_version:
                    return
                self.set_info('Build and release current version anyway until completely in production')
                self.Context.build_task_id = self.Parameters.last_released_data.task_id
                self.set_info('Build Hollywood Fast Data: {}'.format(lb.task_link(self.Context.build_task_id)), do_escape=False)
            else:
                self.Context.build_task_id = BuildHollywoodFastData(
                    self,
                    description='Hollywood Fast Data ({shard}) r{revision}'.format(
                        shard=self.Parameters.shard.upper(), revision=self.Context.revision
                    ),
                    arcadia_url=svn.Arcadia.trunk_url(revision=self.Context.revision),
                    shard=self.Parameters.shard,
                ).enqueue().id
                self.set_info('Build Hollywood Fast Data: {}'.format(lb.task_link(self.Context.build_task_id)), do_escape=False)
            raise sdk2.WaitTask(self.Context.build_task_id, Status.Group.FINISH | Status.Group.BREAK)

        with self.memoize_stage.check_only_build:
            if self.Parameters.only_build:
                self.set_info('Exiting due to "only_build" parameter')
                return

        with self.memoize_stage.check_build:
            build_task = sdk2.Task[self.Context.build_task_id]
            if build_task.status not in Status.Group.SUCCEED:
                self.set_info('Failed to build Hollywood Fast Data on {} revision: {}'.format(
                    self.Context.revision, lb.task_link(self.Context.build_task_id)
                ), do_escape=False)
                raise errors.TaskFailure('Failed to build')
            self.Parameters.fast_data = build_task.Parameters.data_resource
            self.Parameters.fast_data_bundle = build_task.Parameters.bundle_resource
            self.set_info('New Hollywood Fast Data version: {}'.format(self.Parameters.fast_data.version))
            self.Parameters.description += ' ({shard}, release v.{version})'.format(
                shard=self.Parameters.shard.upper(), version=self.Parameters.fast_data.version
            )

        if deploy_settings[self.Parameters.shard].get("run_tests", False) and (not self.Parameters.skip_tests or not self.Parameters.i_accept):
            if self.Context.hamster_deploy_config is not None:
                with self.memoize_stage.start_deploy_on_hamster:
                    self.Context.hamster_deployment_tasks = self.deploy(self.Parameters.fast_data_bundle, deploy_config=self.Context.hamster_deploy_config, deployer_mode='standalone')
                    raise sdk2.WaitTask(self.Context.hamster_deployment_tasks, Status.Group.FINISH | Status.Group.BREAK)

                with self.memoize_stage.check_hamster_deploy:
                    self.Parameters.successfully_deployed_on_hamster = self.check_deploy(self.Context.hamster_deployment_tasks)

                if not self.Parameters.successfully_deployed_on_hamster:
                    with self.memoize_stage.hamster_rollback:
                        self.Context.rollback_hamster_tasks = self.rollback(deploy_config=self.Context.hamster_deploy_config)
                        raise sdk2.WaitTask(self.Context.rollback_hamster_tasks, Status.Group.FINISH | Status.Group.BREAK)

                    with self.memoize_stage.check_hamster_rollback:
                        self.Parameters.successfully_rolled_back_hamster = self.check_rollback(self.Context.rollback_hamster_tasks)

                    raise errors.TaskFailure('Failed to deploy new fast data on hamster. Rollback hamster to previous release {}'.format(
                        'succeeded' if self.Parameters.successfully_rolled_back_hamster else 'failed'
                    ))

            with self.memoize_stage.test_fast_data:
                test_task_id = self.test_fast_data()
                self.Context.test_task_id = test_task_id
                self.set_info('Test Hollywood Fast Data: {}'.format(lb.task_link(test_task_id)), do_escape=False)
                raise sdk2.WaitTask(test_task_id, Status.Group.FINISH | Status.Group.BREAK)

            with self.memoize_stage.check_test_fast_data(commit_on_entrance=False):
                test_task = sdk2.Task[self.Context.test_task_id]
                if test_task.status not in Status.Group.SUCCEED:
                    self.set_info('Tests failed for Hollywood Fast Data on {} revision: {}'.format(
                        self.Context.revision, lb.task_link(self.Context.test_task_id)
                    ), do_escape=False)

                    if self.Context.hamster_deploy_config is not None:
                        with self.memoize_stage.hamster_rollback:
                            self.Context.rollback_hamster_tasks = self.rollback(deploy_config=self.Context.hamster_deploy_config)
                            raise sdk2.WaitTask(self.Context.rollback_hamster_tasks, Status.Group.FINISH | Status.Group.BREAK)

                        with self.memoize_stage.check_hamster_rollback:
                            self.Parameters.successfully_rolled_back_hamster = self.check_rollback(self.Context.rollback_hamster_tasks)

                        raise errors.TaskFailure('Tests failed for new fast data. Rollback hamster to previous release {}'.format(
                            'succeeded' if self.Parameters.successfully_rolled_back_hamster else 'failed'
                        ))
                    else:
                        raise errors.TaskFailure('Tests failed for new fast data.')
                self.set_info('Hollywood Fast Data tested')

        with self.memoize_stage.start_deploy_on_nanny_service:
            if self.Parameters.deployer_mode == 'nanny_service':
                self.Context.deployment_tasks = self.deploy(
                    self.Parameters.fast_data_bundle,
                    deploy_config=self.Context.deploy_config,
                    deployer_mode=self.Parameters.deployer_mode,
                )

        with self.memoize_stage.release:
            self.server.release(
                task_id=self.Context.build_task_id,
                type=ReleaseStatus.STABLE,
                subject="Hollywood Fast Data auto-release"
            )

        with self.memoize_stage.deploy:
            if not self.Context.deployment_tasks:
                self.Context.deployment_tasks = self.deploy(self.Parameters.fast_data_bundle, deploy_config=self.Context.deploy_config, deployer_mode=self.Parameters.deployer_mode)
            raise sdk2.WaitTask(self.Context.deployment_tasks, Status.Group.FINISH | Status.Group.BREAK)

        with self.memoize_stage.check_deploy:
            self.Parameters.successfully_deployed = self.check_deploy(self.Context.deployment_tasks)

        if not self.Parameters.successfully_deployed:
            with self.memoize_stage.rollback:
                self.Context.rollback_tasks = self.rollback(deploy_config=self.Context.deploy_config)
                raise sdk2.WaitTask(self.Context.rollback_tasks, Status.Group.FINISH | Status.Group.BREAK)

            with self.memoize_stage.check_rollback:
                self.Parameters.successfully_rolled_back = self.check_rollback(self.Context.rollback_tasks)

            raise errors.TaskFailure('Failed to deploy new fast data. Rollback to previous release {}'.format(
                'succeeded' if self.Parameters.successfully_rolled_back else 'failed'
            ))

        if self.Parameters.last_released_data is not None and self.Parameters.last_released_data.task_id != self.id:
            sdk2.Task[self.Parameters.last_released_data.task_id].mark_released_resources(ReleaseStatus.STABLE, ttl=90)

    def deploy_new(self, fast_data_bundle, deploy_config, action='activate', deployer_mode=None):
        self.set_info('{} {} (version {})'.format(
            action.capitalize(),
            lb.resource_link(fast_data_bundle.id, 'Hollywood Fast Data'),
            fast_data_bundle.version
        ), do_escape=False)

        services = ', '.join(deploy_config.get('services', {}).keys())
        if not services:
            self.set_info('No services to {} on'.format(action))
            return []

        deployment_task = DeployHollywoodFastData(
            self,
            description='Hollywood Fast Data ({shard}) auto-release for {services}'.format(
                shard=self.Parameters.shard.upper(), services=services,
            ),
            shard=self.Parameters.shard,
            fast_data_bundle=fast_data_bundle,
            deployer_mode=deployer_mode,
            use_testing_deployer=True,
            nanny_service=self.Parameters.nanny_service,
            only_prepare=action == 'prepare',
            deploy_config=deploy_config,
            yt_token_name='robot-voiceint_yt_token',
            nanny_token_name='robot-voiceint_nanny_token',
        ).enqueue()
        self.Context.service_group_name_by_task_id[str(deployment_task.id)] = services

        return [deployment_task.id]

    def deploy(self, fast_data_bundle, deploy_config=None, rollback=False, deployer_mode=None):
        if rollback:
            self.set_info('Rollback to {}'.format(
                lb.resource_link(fast_data_bundle.id, 'v.{}:r{}'.format(fast_data_bundle.version, fast_data_bundle.revision))
            ), do_escape=False)

        self.set_info('Deploy {} (version {})'.format(
            lb.resource_link(fast_data_bundle.id, 'Hollywood Fast Data'),
            fast_data_bundle.version
        ), do_escape=False)

        deployment_tasks = []
        deployment_tasks.extend(self.deploy_new(fast_data_bundle, deploy_config, action='activate', deployer_mode=deployer_mode))

        self.set_info('Deployment started on {}'.format(
            ', '.join(self._get_task_links(deployment_tasks))
        ), do_escape=False)

        return deployment_tasks

    def check_deploy(self, deployment_tasks):
        if not deployment_tasks:
            return True
        succeeded_deployments = set(filter(
            lambda task_id: sdk2.Task[task_id].status in Status.Group.SUCCEED, deployment_tasks
        ))
        if succeeded_deployments:
            self.set_info('Deployment successfully finished on {}'.format(
                ', '.join(self._get_task_links(succeeded_deployments))
            ), do_escape=False)

        failed_deployments = set(deployment_tasks) - succeeded_deployments
        if failed_deployments:
            self.set_info('Deployment failed on {}'.format(
                ', '.join(self._get_task_links(failed_deployments))
            ), do_escape=False)
        return len(failed_deployments) == 0

    def test_fast_data(self):
        task_id = BuildHollywoodFastData(
            self,
            description='Hollywood Fast Data ({shard}) r{revision}'.format(
                shard=self.Parameters.shard.upper(), revision=self.Context.revision
            ),
            arcadia_url=svn.Arcadia.trunk_url(revision=self.Context.revision),
            shard=self.Parameters.shard,
            run_tests=True,
            check_size=False,
        ).enqueue().id
        return task_id

    def rollback(self, deploy_config=None):
        self.server.release(
            task_id=self.Context.build_task_id,
            type=ReleaseStatus.CANCELLED,
            subject="Hollywood Fast Data auto-release"
        )
        if self.Parameters.last_released_data is None:
            self.Parameters.last_released_data = self.Parameters.fast_data_bundle
        return self.deploy(self.Parameters.last_released_data, deploy_config=deploy_config, rollback=True)

    def check_rollback(self, rollback_tasks):
        successfully_rolled_back = self.check_deploy(rollback_tasks)
        if not successfully_rolled_back:
            self.set_info('Failed to rollback')
        return successfully_rolled_back

    def _get_task_links(self, task_ids):
        return map(lambda task_id: lb.task_link(task_id, self.Context.service_group_name_by_task_id[str(task_id)]), task_ids)
