# -*- coding: utf-8 -*-

import logging
import os
import shutil
import tarfile
import tempfile
from contextlib import contextmanager

from sandbox import sdk2
from sandbox.common import errors
from sandbox.sdk2 import svn

from sandbox.projects.common import constants as consts
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.build import parameters as build_parameters
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.noapacheupper import build as nb

from sandbox.projects.hollywood.common import resources
from sandbox.projects.hollywood.common.const import HollywoodConsts
from sandbox.projects.websearch.upper.fast_data.ExecutionTimeTracker import ExecutionTimeTracker

TESTENV_TAG_PREFIX = 'TESTENV-DATABASE'


def add_file_to_tar(bundle_name, path, name):
    with tarfile.open(bundle_name, 'w:gz') as tar:
        tar.add(path, name)


def get_arcadia_build_path(shard):
    if shard == HollywoodConsts.COMMON_SHARD:
        return "alice/hollywood/shards/common/prod/fast_data"
    elif shard == HollywoodConsts.GENERAL_CONVERSATION_SHARD:
        return "alice/hollywood/shards/general_conversation/prod/fast_data"
    else:
        return "alice/hollywood/shards/common/prod/fast_data"


class BuildHollywoodFastData(nanny.ReleaseToNannyTask2, ExecutionTimeTracker):
    _MAX_HOLLYWOOD_FAST_DATA_SIZE = 50

    class Requirements(sdk2.Task.Requirements):
        disk_space = 20 * 1024  # Mb
        cores = 56

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 120 * 60  # 120 min

        arcadia_url = sdk2.parameters.ArcadiaUrl('Svn url for arcadia', required=True)
        check_size = sdk2.parameters.Bool('Check data size', default=True)

        with sdk2.parameters.RadioGroup('Shard') as shard:
            shard.values[HollywoodConsts.COMMON_SHARD] = shard.Value(HollywoodConsts.COMMON_SHARD, default=True)
            shard.values[HollywoodConsts.GENERAL_CONVERSATION_SHARD] = shard.Value(HollywoodConsts.GENERAL_CONVERSATION_SHARD)

        build_system = build_parameters.BuildSystem()

        run_tests = sdk2.parameters.Bool('Run tests', default=False)

        with sdk2.parameters.Output:
            data_resource = sdk2.parameters.Resource('Hollywood Fast Data', resource_type=resources.HollywoodFastData)
            bundle_resource = sdk2.parameters.Resource('Hollywood Fast Data bundle', resource_type=resources.HollywoodFastDataBundle)

    class Context(ExecutionTimeTracker.Context):
        data_name = 'hollywood_fast_data'
        bundle_name = data_name + '.tar.gz'

    @property
    def stage_name(self):
        return 'build'

    def on_execute(self):
        with self.memoize_stage.prepare:
            self.Context.svn_info = svn.Arcadia.info(self.Parameters.arcadia_url)
            self.Context.revision = self.Context.svn_info['commit_revision']
            self.Context.svn_info['sandbox_task'] = self.id
            self.Context.svn_info['shard'] = self.Parameters.shard
            self.Context.version = self._get_version()
            self.Context.svn_info['fast_data_version'] = self.Context.version
            self.Parameters.description += ' v.{}'.format(self.Context.version)

            self.Parameters.data_resource = resources.HollywoodFastData(
                self, self.Parameters.description, self.Context.data_name, ttl=90,
                revision=self.Context.revision, version=self.Context.version, shard=self.Parameters.shard,
            )
            self.Parameters.bundle_resource = resources.HollywoodFastDataBundle(
                self, self.Parameters.description, self.Context.bundle_name, ttl=90,
                revision=self.Context.revision, version=self.Context.version, shard=self.Parameters.shard,
            )

        with self.memoize_stage.build_fast_data:
            data_path = str(self.Parameters.data_resource.path)
            data_name = os.path.basename(data_path)

            self.build(data_path)
            self.add_svn_info_file(data_path)
            nb.tar(self.Context.bundle_name, data_path, data_name)

            self.Context.size = nb.get_directory_size(data_path)
            self.set_info('Size: {0:.1f}Mb'.format(self.Context.size))

            if self.Parameters.check_size:
                if self.Context.size > self._MAX_HOLLYWOOD_FAST_DATA_SIZE:
                    raise errors.TaskFailure('Data size exceeds maximum allowed size {}Mb'.format(self._MAX_HOLLYWOOD_FAST_DATA_SIZE))

    @contextmanager
    def get_arcadia(self):
        try:
            if not sdk.fuse_available():
                raise errors.TaskFailure('Fuse unavailable')
            with sdk.mount_arc_path(self.Parameters.arcadia_url) as arcadia:
                yield arcadia
        except errors.TaskFailure as e:
            logging.exception(e)
            yield svn.Arcadia.get_arcadia_src_dir(self.Parameters.arcadia_url)

    def build(self, data_path):
        target = get_arcadia_build_path(self.Parameters.shard)
        build_dir = tempfile.mkdtemp()

        with self.get_arcadia() as arcadia:
            logging.info('Build hollywood_fast_data...')
            returncode = sdk.do_build(
                self.Parameters.build_system,
                build_type=consts.RELEASE_BUILD_TYPE,
                clear_build=True,
                results_dir=build_dir,
                source_root=arcadia,
                targets=[target],
                test=self.Parameters.run_tests,
            )
            if returncode != 0:
                if self.Parameters.run_tests:
                    logging.info('hollywood_fast_data tests failed, returncode: {}'.format(returncode))
                    raise errors.TaskFailure('hollywood_fast_data tests failed, returncode: {}'.format(returncode))
                else:
                    logging.info('Failed to build hollywood_fast_data, returncode: {}'.format(returncode))
                    raise errors.TaskFailure('Failed to build hollywood_fast_data, returncode: {}'.format(returncode))

            if self.Parameters.run_tests:
                logging.info('hollywood_fast_data tested')
            else:
                logging.info('hollywood_fast_data built')

        shutil.move(os.path.join(build_dir, target), data_path)

    def add_svn_info_file(self, data_path):
        logging.info('write .svninfo...')
        with open(os.path.join(data_path, '.svninfo'), 'w') as svn_info_file:
            svn_info_file.write('\n'.join(
                '{}: {}'.format(k, v) for k, v in sorted(self.Context.svn_info.iteritems())
            ) + '\n')

    def _get_version(self):
        current_revision_resource = resources.HollywoodFastData.find(
            attrs={'revision': self.Context.revision, 'shard': self.Parameters.shard},
        ).order(-sdk2.Resource.accessed).first()
        if current_revision_resource and current_revision_resource.version:
            logging.debug('Found resource with this revision: {}'.format(current_revision_resource.id))
            return current_revision_resource.version

        resources_list = list(resources.HollywoodFastData.find(attrs={'shard': self.Parameters.shard}).order(-sdk2.Resource.accessed).limit(25))
        if resources_list:
            max_resource_version = max(map(lambda resource: resource.version, resources_list))
            return max_resource_version + 1
        return 1

    def on_release(self, additional_params):
        if any((tag.startswith(TESTENV_TAG_PREFIX) for tag in self.Parameters.tags)):
            nanny.ReleaseToNannyTask2.on_release(self, additional_params)
        sdk2.Task.on_release(self, additional_params)
