# -*- coding: utf-8 -*-

import json
import logging
import os
import time

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types.resource import State
from sandbox.common.types.task import ReleaseStatus
from sandbox.common.types.task import Semaphores
from sandbox.projects.common import decorators
from sandbox.projects.common.nanny.client import NannyClient
from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess

from sandbox.projects.hollywood.common import resources
from sandbox.projects.hollywood.common.const import HollywoodConsts

from sandbox.projects.websearch.params import ResourceWithLastReleasedValueByDefault
from sandbox.projects.websearch.upper import resources as websearch_upper_resources
from sandbox.projects.websearch.upper.fast_data.ExecutionTimeTracker import ExecutionTimeTracker


NANNY_API_URL = 'http://nanny.yandex-team.ru/'


def get_deploy_config_type(shard):
    if shard == HollywoodConsts.COMMON_SHARD:
        return resources.HollywoodCommonFastDataDeployConfig
    elif shard == HollywoodConsts.GENERAL_CONVERSATION_SHARD:
        return resources.HollywoodGeneralConversationFastDataDeployConfig
    else:
        return resources.HollywoodFastDataDeployConfig


class DeployHollywoodFastData(ExecutionTimeTracker):
    class Requirements(sdk2.Task.Requirements):
        cores = 8
        environments = [
            environments.PipEnvironment('yandex-yt'),
        ]

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600  # 1 hour

        fast_data_bundle = sdk2.parameters.Resource(
            "Fast Data bundle to deploy",
            resource_type=[
                resources.HollywoodFastDataBundle,
            ],
            required=True,
        )

        with sdk2.parameters.RadioGroup('Shard') as shard:
            shard.values[HollywoodConsts.COMMON_SHARD] = shard.Value(HollywoodConsts.COMMON_SHARD, default=True)
            shard.values[HollywoodConsts.GENERAL_CONVERSATION_SHARD] = shard.Value(HollywoodConsts.GENERAL_CONVERSATION_SHARD)

        with sdk2.parameters.RadioGroup("Deployer mode") as deployer_mode:
            deployer_mode.values['standalone'] = deployer_mode.Value('Run deployer in task', default=True)
            deployer_mode.values['nanny_service'] = deployer_mode.Value('Deploy config to nanny service with deployer')
            with deployer_mode.value['standalone']:
                use_testing_deployer = sdk2.parameters.Bool("Use testing deployer", default=False)
                with use_testing_deployer.value[False]:
                    deployer = ResourceWithLastReleasedValueByDefault(
                        "search/tools/fast_data_deployment/deployer binary",
                        resource_type=websearch_upper_resources.FastDataDeployer,
                        required=True,
                    )
            with deployer_mode.value['nanny_service']:
                nanny_service = sdk2.parameters.String('Deployer Nanny service name', required=True)

        only_prepare = sdk2.parameters.Bool('Just prepare data, do not activate', default=False)
        deploy_config = sdk2.parameters.JSON('Deploy config', required=True)

        with sdk2.parameters.Group('Vault'):
            yt_token_name = sdk2.parameters.String('YT token name', required=True, default='robot-voiceint_yt_token')
            infra_token_owner = sdk2.parameters.String('INFRA_TOKEN owner')
            solomon_token_owner = sdk2.parameters.String('SOLOMON_TOKEN owner')
            nanny_token_name = sdk2.parameters.String('Nanny token name', required=True, default='robot-voiceint_nanny_token')

        with sdk2.parameters.Output:
            deploy_config_output_resource = sdk2.parameters.Resource("Deploy config", resource_type=resources.HollywoodFastDataDeployConfig)

    @property
    def stage_name(self):
        return 'deploy'

    def on_enqueue(self):
        super(DeployHollywoodFastData, self).on_enqueue()
        if not self.Requirements.semaphores:
            self.Requirements.semaphores = Semaphores(
                acquires=[
                    Semaphores.Acquire(
                        name='lock-{}_{}-deployer'.format(service, self.Parameters.deployer_mode),
                        weight=2,
                        capacity=2
                    )
                    for service in self.Parameters.deploy_config["services"].keys()
                ],
            )

    def on_execute(self):
        with self.memoize_stage.patch_parameters:
            self.Context.deploy_config = self.Parameters.deploy_config
            if self.Parameters.fast_data_bundle:
                self.Context.deploy_config.setdefault("resource", {}).update({
                    "url": self.Parameters.fast_data_bundle.skynet_id,
                    "version": self.Parameters.fast_data_bundle.version,
                })

            self.Context.deploy_config.setdefault("deploy", {}).update({
                "only_prepare": self.Parameters.only_prepare,
            })

        with self.memoize_stage.save_config_resource:
            config_path = 'deploy_config.json'
            self.save_deploy_config_resource(self.Context.deploy_config, config_path)
            self.mark_released_resources('stable', ttl=90)

        if self.Parameters.deployer_mode == 'nanny_service':
            self.run_on_nanny_service(self.Context.deploy_config, self.Parameters.nanny_service)
        elif self.Parameters.deployer_mode == 'standalone':
            self.run_deployer(config_path)
        else:
            raise errors.TaskFailure('Unknown deploy mode')

    def save_deploy_config_resource(self, deploy_config, config_path):
        self.Parameters.deploy_config_output_resource = get_deploy_config_type(self.Parameters.shard)(
            self, self.Parameters.description, config_path, ttl=90
        )
        with open(config_path, 'w') as config_fd:
            json.dump(deploy_config, config_fd, indent=2)
        sdk2.ResourceData(self.Parameters.deploy_config_output_resource).ready()
        return self.Parameters.deploy_config_output_resource

    def run_on_nanny_service(self, deploy_config, nanny_service):
        from yt import wrapper as yw

        nanny_client = NannyClient(NANNY_API_URL, sdk2.Vault.data(self.Parameters.nanny_token_name))

        with self.memoize_stage.prepare_config:
            logging.info('Update deploy config resource for {}'.format(nanny_service))
            data = nanny_client.update_service_sandbox_file(
                service_id=nanny_service,
                task_type=str(self.type),
                task_id=str(self.id),
                skip_not_existing_resources=False,
                allow_empty_changes=False,
            )
            logging.info('Update file response:\n{}'.format(json.dumps(data, indent=2)))

            self.Context.snapshot_id = data['runtime_attrs']['_id']
            logging.info('Set snapshot {} state to PREPARED'.format(self.Context.snapshot_id))
            nanny_client.set_snapshot_state(
                service_id=nanny_service,
                snapshot_id=self.Context.snapshot_id,
                state='PREPARED',
                comment='Prepare Fast Data release v.{} (task {})'.format(self.Parameters.fast_data_bundle.version, self.id),
                prepare_recipe='default',
                set_as_current=True,
            )

        with self.memoize_stage.activate_config:
            yt_client = yw.YtClient(deploy_config['communication']['cluster'], sdk2.Vault.data(self.Parameters.yt_token_name))

            def get_actions():
                return ['prepare'] + ['activate'] * (not deploy_config['deploy'].get('only_prepare', False))

            def list_status_paths():
                for service_id in deploy_config.get('services', {}).keys():
                    service_cypress_path = yw.ypath_join(deploy_config['communication']['cypress_dir'], service_id)
                    for geo in yt_client.list(service_cypress_path):
                        feedback_path = yw.ypath_join(service_cypress_path, geo, 'feedback')
                        if yt_client.exists(feedback_path):
                            for action in get_actions():
                                yield service_id, action, yw.ypath_join(feedback_path, '@{}'.format(action))

            @decorators.retries(5, delay=3, backoff=1)
            def clear():
                logging.info('Clear status paths...')
                for _, _, status_path in list_status_paths():
                    logging.info('Clear path {}'.format(status_path))
                    yt_client.set(status_path, {})

            def split_by_status():
                failed, succeeded, inprogress = [], [], []
                for service_id, action, status_path in list_status_paths():
                    status = yt_client.get(status_path)
                    logging.info('service {}, action {}, status = {} ({})'.format(service_id, action, status, status_path))
                    if status.get('failed') == deploy_config['resource']['version']:
                        failed.append([service_id, action])
                    elif status.get('current') == deploy_config['resource']['version']:
                        succeeded.append([service_id, action])
                    else:
                        inprogress.append([service_id, action])
                return failed, succeeded, inprogress

            clear()

            logging.info('Set snapshot {} state to ACTIVE'.format(self.Context.snapshot_id))
            nanny_client.set_snapshot_state(
                service_id=nanny_service,
                snapshot_id=self.Context.snapshot_id,
                state='ACTIVE',
                comment='Fast Data release v.{} (task {})'.format(self.Parameters.fast_data_bundle.version, self.id),
                recipe='default',
                set_as_current=True,
            )

            processed = {action: set() for action in get_actions()}
            while True:
                failed, succeeded, inprogress = split_by_status()

                for service_id, action in succeeded:
                    if service_id not in processed[action]:
                        self.set_info('Done {} stage for {}'.format(action.upper(), service_id))
                        processed[action].add(service_id)

                if failed:
                    raise errors.TaskFailure('Failed to {}'.format(', '.join(map(
                        lambda (service_id, action): '{} on {}'.format(action.upper(), service_id),
                        failed
                    ))))

                if len(inprogress) == 0:
                    break

                time.sleep(1)

    def run_deployer(self, config_path):
        if self.Parameters.use_testing_deployer:
            attrs = [
                ["released", ReleaseStatus.STABLE],
                ["released", ReleaseStatus.PRESTABLE],
                ["released", ReleaseStatus.TESTING],
            ]
            deployer_id = sdk2.Task.server.resource.read(
                type=websearch_upper_resources.FastDataDeployer,
                status=State.READY,
                attrs=json.dumps(attrs),
                any_attr=True,
                limit=1
            )["items"][0]["id"]
            deployer = sdk2.Resource[deployer_id]
        else:
            deployer = self.Parameters.deployer

        deployer_path = str(sdk2.ResourceData(deployer).path)

        cmd = [
            'stdbuf', '-o', 'L',
            deployer_path, '--config', config_path,
        ]

        deployer_env = os.environ.copy()
        deployer_env['YT_TOKEN'] = sdk2.Vault.data(self.Parameters.yt_token_name)
        deployer_env['OAUTH_NANNY'] = sdk2.Vault.data(self.Parameters.nanny_token_name)

        if self.Parameters.infra_token_owner:
            deployer_env['INFRA_TOKEN'] = sdk2.Vault.data(self.Parameters.infra_token_owner, 'INFRA_TOKEN')

        deployer_env['SOLOMON_TOKEN'] = sdk2.Vault.data('robot-bassist_solomon_token')
        deployer_env['YT_READ_FROM_CACHE'] = '0'

        self.set_info('Starting deployer:\n{}'.format(' '.join(cmd)))
        with sdk2.helpers.ProcessLog(self, logger="deployer") as pl:
            process = subprocess.Popen(
                cmd,
                env=deployer_env,
                stdin=subprocess.PIPE,
                stdout=subprocess.PIPE,
                stderr=pl.stderr,
                close_fds=True,
            )
            process.stdin.close()
            try:
                while True:
                    line = process.stdout.readline()
                    if not line:
                        break
                    self.set_info(line.strip('\n'))
                result = process.wait()
                self.set_info('Deployer finished')

                if result:
                    raise errors.TaskFailure('Failed to deploy')
            finally:
                process.stdout.close()
        self.set_info('Successfully deployed')
