from sandbox import sdk2
import os
import tempfile
import re
from sandbox.projects.common.utils import sync_last_stable_resource
from sandbox.projects.sandbox.sandbox_lxc_image import SandboxLxcImage
from sandbox.projects.home.resources import PortalHomeBuildTools
from sandbox.sandboxsdk.paths import copy_path
import tarfile

CUSTOM_SCRIPT = ['update-alternatives --install /usr/bin/node node /opt/nodejs/8/bin/node 110 --slave /usr/bin/npm npm /opt/nodejs/8/bin/npm --slave /usr/bin/npx npx /opt/nodejs/8/bin/npx',
                 'npm install -g svgo eslint',
                 'pip install awscli',
                 'apt-add-repository ppa:qameta/allure -y',
                 'apt-get update',
                 'apt-get install -y --force-yes allure']


CUSTOM_PACKAGES = ['yandex-debhelper-all',
                   'yandex-verstka-cdbs',
                   'git',
                   'webp',
                   'debhelper',
                   'xsltproc',
                   'libxml2-utils',
                   'liblib-abs-perl',
                   'nodejs',
                   'nodejs-6',
                   'nodejs-8',
                   'optipng',
                   'dupload',
                   'libconfig-apacheformat-perl',
                   'libtemplate-perl',
                   'buildfarm-pbuilder',
                   'conductor-utils',
                   'python',
                   'python-pip',
                   'python-virtualenv',
                   'software-properties-common',
                   'python-software-properties']


class MordaRepoInfo(object):
    def __init__(self, url, branch='master', projects=None):
        self.url = url
        self.branch = branch
        self.projects = projects if projects else ['/']

    def dir(self, root_dir):
        if ('github.yandex-team.ru' in self.url):
            line = re.sub(
                r"^.*github.yandex-team.ru[:/]",
                '',
                self.url
            )
            return '{}/{}'.format(root_dir, line.replace('.git', ''))
        else:
            return tempfile.mkdtemp()


class BuildHomeBuildagentLxc(SandboxLxcImage):

    REPO_DIR = '/opt/repos'

    class Parameters(SandboxLxcImage.Parameters):
        custom_image = SandboxLxcImage.Parameters.custom_image(default=True)
        custom_script = SandboxLxcImage.Parameters.custom_script(default='\n'.join(CUSTOM_SCRIPT))
        custom_packages = SandboxLxcImage.Parameters.custom_packages(default=' '.join(CUSTOM_PACKAGES))
        custom_attrs = SandboxLxcImage.Parameters.custom_attrs(default=dict(ttl=30, released='stable'))

    def get_repos(self):
        return []

    def clone_repos(self):
        self.execute('mkdir -p {}'.format(self.REPO_DIR), True)
        projects = []

        for repo in self.get_repos():
            path = self.git_clone(repo)
            projects.extend([path + p for p in repo.projects])

        self.Context.projects = projects

    def git_clone(self, repo):
        repo_dir = repo.dir(BuildHomeBuildagentLxc.REPO_DIR)
        with sdk2.ssh.Key(self, "HOME", "home_buildfarm"):
            self.execute('git clone --progress -b {} {} {}'.format(repo.branch, repo.url, repo_dir), True)
        return repo_dir

    def get_build_tools(self):
        tools_path_src = sync_last_stable_resource(PortalHomeBuildTools)
        tools_path_dst = os.path.join(self.rootfs + '/opt', os.path.basename(tools_path_src))
        copy_path(tools_path_src, tools_path_dst)
        tarfile.open(tools_path_dst, 'r').extractall(self.rootfs + '/opt')
        os.remove(tools_path_dst)

    def get_geodata(self):
        geobase_dir = self.rootfs + '/var/cache/geobase'
        from projects.geobase.Geodata4BinStable import resource as gb4r
        from projects.geobase.Geodata5BinStable import resource as gb5r
        for geobase in [gb4r.GEODATA4BIN_STABLE, gb5r.GEODATA5BIN_STABLE]:
            geodata_path_src = sync_last_stable_resource(geobase)
            geodata_path_dst = os.path.join(geobase_dir, os.path.basename(geodata_path_src))
            copy_path(geodata_path_src, geodata_path_dst)
            if geobase == gb4r.GEODATA4BIN_STABLE:
                tarfile.open(geodata_path_dst, 'r').extractall(self.rootfs + '/var/cache/geobase')
                os.remove(geodata_path_dst)

    def get_morda_repos(self):
        return []

    def install_deps(self):
        for p in self.Context.projects:
            self.execute('perl /opt/build/install-deps.pl --control {}/debian/control --mode build_container'.format(p), True)

    def cook_image(self, current_dir):
        SandboxLxcImage.cook_image(self, current_dir)
        self.get_geodata()
        self.get_build_tools()
        self.clone_repos()
        self.install_deps()
