from sandbox import sdk2
import shutil
import os
import tempfile
import logging
import re
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.sdk2.path as spath
from sandbox.projects.common.gnupg import GpgKey
from sandbox.projects.common.debpkg import DebRelease
from sandbox import sandboxsdk
from sandbox.projects.common.nanny import nanny
import sandbox.common.types.resource as ctr
from sandbox.projects.home.common.tools import getPotatoUrl
from sandbox.projects.trendbox_ci.beta.managers.vault import VaultManager


DUPLOAD_CONF = {
    'common': {
        'fqdn': 'common.dupload.dist.yandex.ru',
        'method': 'scpb',
        'incoming': '/repo/common/mini-dinstall/incoming',
        'dinstall_runs': 0,
        'login': 'robot-morda-wan'
    },
    'verstka': {
        'fqdn': 'verstka.dupload.dist.yandex.ru',
        'method': 'scpb',
        'incoming': '/repo/verstka/mini-dinstall/incoming',
        'dinstall_runs': 0,
        'login': 'robot-morda-wan'
    },
    'morda-precise': {
        'fqdn': 'morda-precise.dupload.dist.yandex.ru',
        'method': 'scpb',
        'incoming': '/repo/morda-precise/mini-dinstall/incoming',
        'dinstall_runs': 0,
        'login': 'robot-morda-wan'
    },
    'morda-trusty': {
        'fqdn': 'morda-trusty.dupload.dist.yandex.ru',
        'method': 'scpb',
        'incoming': '/repo/morda-trusty/mini-dinstall/incoming',
        'dinstall_runs': 0,
        'login': 'robot-morda-wan'
    },
    'morda-xenial': {
        'fqdn': 'morda-xenial.dupload.dist.yandex.ru',
        'method': 'scpb',
        'incoming': '/repo/morda-xenial/mini-dinstall/incoming',
        'dinstall_runs': 0,
        'login': 'robot-morda-wan'
    }
}


class MordaResource(object):
    def __init__(self, task, resource_type, description=''):
        self.task = task
        self.resource_type = resource_type
        self.description = description + '\n' if description else description

    def create_resource(self):
        raise Exception('Not implemented create_resource method')


class MordaFileResource(MordaResource):
    def __init__(self, task, resource_type, path, description=''):
        super(MordaFileResource, self).__init__(task, resource_type, description)
        self.path = path

    def create_resource(self):

        resource = self.resource_type(self.task, (self.description + self.task.get_changelog()).replace('\n', '<br/>'), 'resource', version=self.task.get_version())
        resource_data = sdk2.ResourceData(resource)

        resource_data.path.write_bytes(spath.Path(self.path).read_bytes())
        resource_data.ready()


class MordaTarballResource(MordaResource):
    def __init__(self, task, resource_type, file_name='tarball.tar.gz', description=''):
        super(MordaTarballResource, self).__init__(task, resource_type, description)
        self.file_name = file_name

    def prepare_data(self):
        pass

    def create_resource(self):
        path, directory = self.prepare_data()
        tar_command = 'tar zcvf {} -C {} {}/'.format(self.file_name, path, directory)

        with open('{}/{}/{}.version'.format(path, directory, self.task.get_source()), 'w') as version_file:
            version_file.write(self.task.get_version())

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("create_tarball")) as pl:
            status = sp.Popen(tar_command, shell=True, stdout=pl.stdout, stderr=sp.STDOUT, cwd=self.task.Context.root_dir).wait()
            if status != 0:
                raise Exception('Failed to create tarball')

            resource = self.resource_type(self.task, (self.description + '\n' + self.task.get_changelog()).replace('\n', '<br/>'), self.file_name, version=self.task.get_version())
            resource_data = sdk2.ResourceData(resource)
            resource_data.path.write_bytes(spath.Path(self.task.Context.root_dir, self.file_name).read_bytes())
            resource_data.ready()


class HomeDebuilderTask(nanny.ReleaseToNannyTask2, sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        privileged = True

    class Parameters(sdk2.Task.Parameters):
        container = sdk2.parameters.Container("Container", required=True)

        with sdk2.parameters.Group("Git parameters") as git_block:
            git_repo = sdk2.parameters.String('Git repository', required=True)
            branch = sdk2.parameters.String('Branch', required=True, default_value='release')

        project_dir = sdk2.parameters.String('Project directory', required=False)
        create_tarball = sdk2.parameters.Bool('Create tarball', required=True, default=False)

        beta_build = sdk2.parameters.Bool('Build beta package', required=False, default=False)

        with create_tarball.value[True]:
            force_rebuild = sdk2.parameters.Bool('Rebuild', required=True, default=True)
            with sdk2.parameters.String('Auto release') as auto_release:
                auto_release.values[""] = "-"
                for status in [ctt.ReleaseStatus.TESTING, ctt.ReleaseStatus.UNSTABLE, ctt.ReleaseStatus.PRESTABLE, ctt.ReleaseStatus.STABLE]:
                    auto_release.values[status] = status

        make_debuild = sdk2.parameters.Bool('Make debuild', required=True, default=True)

        with sdk2.parameters.Group('Nanny webhooks') as hooks:
            webhook_urls = nanny.WebHookUrlParameter2('Urls')
            webhook_type = nanny.WebHookTypeParameter2('Webhook type', default_value='RELEASE_ONLY')

        with make_debuild.value[True]:
            with sdk2.parameters.Group("Dist parameters") as dist_block:
                to_dupload = sdk2.parameters.Bool('Dupload', required=True, default_value=True)
                with to_dupload.value[True]:
                    dist_repo = sdk2.parameters.String('Repository', required=True)
                    debticket = sdk2.parameters.Bool('Create conductor tickets', required=True, default=True)

    class Context(sdk2.Task.Context):
        pass

    def dir(self, url):
        if ('git@github.yandex-team.ru' in url):
            return '/opt/repos/{}'.format(url.split(':')[1].replace('.git', ''))

    def git_clone(self):
        cached_repo_dir = self.dir(self.Parameters.git_repo)

        if cached_repo_dir and os.path.exists(cached_repo_dir):
            repo_dir = cached_repo_dir
            root_dir = repo_dir + '/..'
            with sandboxsdk.ssh.Key(self, "HOME", "home_buildfarm"):
                with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("git_checkout")) as pl:
                    status = sp.Popen('git fetch --tags && git checkout {}'.format(self.Parameters.branch),
                                      shell=True,
                                      stdout=pl.stdout,
                                      stderr=sp.STDOUT,
                                      cwd=repo_dir).wait()
                    if status != 0:
                        raise Exception('Failed to checkout {}'.format(self.Parameters.branch))
        else:
            repo_dir = tempfile.mkdtemp()
            root_dir = repo_dir + '/..'

            with sandboxsdk.ssh.Key(self, "HOME", "home_buildfarm"):
                with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("git_checkout")) as pl:
                    status = sp.Popen('git clone --progress --depth 1 -b {} {} {}'.format(self.Parameters.branch, self.Parameters.git_repo, repo_dir),
                                      shell=True,
                                      stdout=pl.stdout,
                                      stderr=sp.STDOUT,
                                      cwd=repo_dir).wait()
                    if status != 0:
                        raise Exception('Failed to clone {}'.format(self.Parameters.git_repo))

        project_dir = repo_dir
        if self.Parameters.project_dir:
            project_dir += '/' + self.Parameters.project_dir

        self.Context.root_dir = root_dir
        self.Context.repo_dir = repo_dir
        self.Context.project_dir = project_dir
        self.Context.deb_dir = project_dir + '/..'

    def prepare_deps(self):
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("prepare_deps")) as pl:
            status = sp.Popen('perl /opt/build/install-deps.pl --control {}/debian/control --mode debuild'.format(self.Context.project_dir),
                              shell=True,
                              stdout=pl.stdout,
                              stderr=sp.STDOUT,
                              cwd=self.Context.project_dir).wait()
            if status != 0:
                raise Exception('Failed to install dependencies')

    def get_debuild_params(self):
        return ['--no-tgz-check',
                '--no-lintian',
                '--preserve-env',
                '-b',
                '-krobot-morda-wan@yandex-team.ru',
                '-sa']

    def before_build(self):
        pass

    def debuild(self):
        self.prepare_deps()
        debuild_command = ['debuild'] + self.get_debuild_params()

        vault_manager = VaultManager(self)
        vault_envs = vault_manager.get_all_vault_env().items()

        with GpgKey(sandboxsdk.channel.channel.task, "HOME", "home_buildfarm_gpg_private", "home_buildfarm_gpg_public"):
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("debuild")) as pl:
                status = sp.Popen(' '.join(debuild_command),
                                  shell=True,
                                  stdout=pl.stdout,
                                  stderr=sp.STDOUT,
                                  cwd=self.Context.project_dir,
                                  env=dict(os.environ.items() + vault_envs)).wait()
                if status != 0:
                    raise Exception('Failed to build deb')

    def dupload(self, repository, changes_file):
        dupload_params = ['dupload',
                          '--to',
                          repository,
                          '--nomail',
                          changes_file]

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("dupload")) as pl:
            with DebRelease(DUPLOAD_CONF):
                with sandboxsdk.ssh.Key(self, "HOME", "home_buildfarm"):
                    status = sp.Popen(' '.join(dupload_params),
                                      shell=True,
                                      stdout=pl.stdout,
                                      stderr=sp.STDOUT,
                                      cwd=self.Context.deb_dir).wait()
                    if status != 0:
                        raise Exception('Failed to dupload')

    def get_version(self):
        if not self.Context.version:
            if self.Parameters.beta_build:
                self.Context.version = self.get_git_hash()
            else:
                self.Context.version = self.get_deb_version()

        return self.Context.version

    def get_deb_version(self):
        if not self.Context.deb_version:
            self.Context.deb_version = sp.check_output(["head -n 1 debian/changelog | sed -e 's/.*[(]//' | sed -e 's/[)].*//'"],
                                                        shell=True,
                                                        cwd=self.Context.project_dir).strip()

        return self.Context.deb_version

    def get_git_hash(self):
        if not self.Context.git_hash:
            self.Context.git_hash = sp.check_output(['git show --pretty=format:"%h" -s'],
                                                    shell=True,
                                                    cwd=self.Context.project_dir).strip()
        return self.Context.git_hash

    def get_source(self):
        if not self.Context.source:
            self.Context.source = sp.check_output(["head -n 1 debian/changelog | awk '{print $1}'"],
                                                  shell=True,
                                                  cwd=self.Context.project_dir).strip()
        return self.Context.source

    def get_arch(self):
        if not self.Context.arch:
            self.Context.arch = sp.check_output(["dpkg-architecture | grep DEB_BUILD_ARCH= | sed -e 's/.*=//'"],
                                                shell=True,
                                                cwd=self.Context.project_dir).strip()
        return self.Context.arch

    def get_changelog(self):
        if not self.Context.changelog:
            self.Context.changelog = sp.check_output("dpkg-parsechangelog | sed -n '/Changes:/,$p' | tail -n +4 | sed 's/^..//'",
                                                     shell=True,
                                                     cwd=self.Context.project_dir).strip()
        return self.Context.changelog

    def get_startrek_tickets(self):
        if not self.Context.startrek_ticket_ids:
            changelog = self.get_changelog()
            exclude = set(['HOME-0', 'HOME-1', 'ETHER-0'])

            tickets = re.findall('\\b([A-Z]+-\\d+)', changelog, re.MULTILINE)
            self.Context.startrek_ticket_ids = [ticket for ticket in tickets if ticket not in exclude]

        return self.Context.startrek_ticket_ids

    def get_pkg_changes_file(self):
        source = self.get_source()
        version = self.get_version()
        arch = self.get_arch()
        pkg_changes = '{}_{}_{}.changes'.format(source, version, arch)

        return pkg_changes

    def get_resources(self):
        return []

    def create_tarball(self, resource_type, path, directory, file_name, description=None):
        tar_command = 'tar zcvf {} -C {} {}/'.format(file_name, path, directory)

        with open('{}/{}/{}.version'.format(path, directory, self.get_source()), 'w') as version_file:
            version_file.write(self.get_version())

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("create_tarball")) as pl:
            status = sp.Popen(tar_command, shell=True, stdout=pl.stdout, stderr=sp.STDOUT, cwd=self.Context.root_dir).wait()
            if status != 0:
                raise Exception('Failed to create tarball')

            resource = resource_type(self, (description + '\n' + self.get_changelog()).replace('\n', '<br/>'), file_name, version=self.get_version())
            resource_data = sdk2.ResourceData(resource)
            resource_data.path.write_bytes(spath.Path(self.Context.root_dir, file_name).read_bytes())
            resource_data.ready()

    def clone_built_resources(self, resource_type, version):
        if not (resource_type and version):
            return

        logging.info("Trying to find already built resource {}={}".format(resource_type, version))
        resource = sdk2.Resource.find(resource_type,
                                      attrs=dict(version=version),
                                      state=[ctr.State.READY]
                                      ).first()
        if not resource:
            logging.info("No resource found")
            return False

        try:
            resource_data = sdk2.ResourceData(resource)
            logging.info("Found resource {}. Cloning to current task".format(str(resource)))
            resource_path = str(resource_data.path)
            clone_dir_path = str(self.path('{}.{}'.format(self.id, resource.id)))
            os.mkdir(clone_dir_path)

            if os.path.isdir(resource_path):
                shutil.copytree(resource_path, os.path.join(clone_dir_path, os.path.basename(resource_path)))
            elif os.path.isfile(resource_path):
                shutil.copy2(resource_path, clone_dir_path)
            else:
                raise RuntimeError('not a file or directory: {} (task={}, resource={})'.format(resource_path, self.id, resource.id))

            clone_path = os.path.join(clone_dir_path, os.path.basename(resource_path))

            resource_type = type(resource)

            sdk2.ResourceData(resource_type(self, resource.description, clone_path, version=resource.version))
            logging.info('Resource {} successfully cloned'.format(str(resource)))
            return True
        except Exception as e:
            logging.info('Failed to clone resource {}'.format(resource))
            logging.exception(e)
            return False

    def create_resource_data(self):
        raise Exception('Not implemented create_resource_data method')

    def get_conductor_packages(self):
        return ''

    def create_conductor_tickets(self):
        if not self.Parameters.debticket:
            return

        token = sdk2.Vault.data('HOME', 'home_conductor')
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("create_conductor_tickets")) as pl:
            status = sp.Popen('debticket -n --oauth ' + token + ' ' + self.get_conductor_packages(),
                              shell=True,
                              stdout=pl.stdout,
                              stderr=sp.STDOUT,
                              cwd=self.Context.project_dir).wait()
            if status != 0:
                raise Exception('Failed to create conductor tickets')

    def set_package_info(self):
        changelog = self.get_changelog()
        source = self.get_source()
        version = self.get_version()
        tickets = self.get_startrek_tickets()
        description = "{}={}\n{}\n{}".format(source, version, changelog, '\n'.join(tickets))
        self.Parameters.description = description

    def on_save(self):
        urls = self.Parameters.webhook_urls

        if not urls:
            repo = re.search('([\\w-]+/[\\w-]+).git$', self.Parameters.git_repo)

            if repo:
                self.Parameters.webhook_urls.append(getPotatoUrl(repo.group(1)))


    def on_execute(self):
        with self.memoize_stage.git_clone:
            self.git_clone()

        self.before_build()
        self.set_package_info()

        if not self.Parameters.force_rebuild:
            cloned = all([self.clone_built_resources(resource.resource_type, self.get_version()) for resource in self.get_resources()])
            if cloned:
                self.Context.cloned = True
                logging.info('Cloned resource. Finishing and skipping debuild')
                return

        if self.Parameters.make_debuild:
            with sandboxsdk.ssh.Key(self, "HOME", "home_buildfarm"):
                self.debuild()

            if self.Parameters.to_dupload:
                pkg_changes = self.get_pkg_changes_file()
                self.dupload(self.Parameters.dist_repo, pkg_changes)
                self.create_conductor_tickets()

        if self.Parameters.create_tarball:
            for resource in self.get_resources():
                resource.create_resource()

    def on_success(self, prev_status):
        sdk2.Task.on_success(self, prev_status)
        if self.Parameters.create_tarball and self.Parameters.auto_release:
            self.on_release(dict(release_status=self.Parameters.auto_release))

    def on_release(self, additional_parameters):
        resources = []

        for resource in sdk2.Resource.find(task=self, state=ctr.State.READY).limit(0):
            if not resource.type.releasable:
                continue
            resources.append(resource)

        release_subject = ', '.join(['{}={}'.format(r.type.name, r.version) for r in resources])
        additional_parameters['release_subject'] = release_subject
        additional_parameters['release_comments'] = self.get_changelog()
        logging.info(str(additional_parameters))
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
